/* aout.c - parse and load the contents of a UNIX a.out file, for
 * several flavours of PDP-11 UNIX
 */
#include "defines.h"
#define EIGHT_K		 8192

/* UNIX magic numbers for the a.out header */
#define ANY_NORMAL	0407	/* normal: V5,V6,V7,2.11BSD */
#define ANY_ROTEXT	0410	/* read-only text: V5,V6,V7,2.11BSD */
#define ANY_SPLITID	0411	/* seperated I&D: V5,V6,V7,2.11BSD */
#define ANY_OVERLAY	0405	/* overlay: V5,V6,V7,2.11BSD */
#define BSD_OVERLAY	0430	/* 2.11BSD overlay, non-separate */
#define BSD_ROVERLAY	0431	/* 2.11BSD overlay, separate */

/* a.out header for all UNIX flavours */
struct exec {
    uint16_t a_magic;		/* magic number */
    uint16_t a_text;		/* size of text segment */
    uint16_t a_data;		/* size of initialised data */
    uint16_t a_bss;		/* size of initialised bss */
    uint16_t a_syms;		/* size of symbol table */
    uint16_t a_entry;		/* entry point */
    uint16_t a_unused;		/* unused */
    uint16_t a_flag;		/* relocation info stripped */
				/* 16 bytes up to here */

				/* 2.11BSD overlay files have the following */
#define NOVL	15
     int16_t max_ovl;		/* maximum overlay size */
    uint16_t ov_siz[NOVL];	/* size of the i'th overlay */
				/* Note that if the file isn't a 2.11BSD */
				/* overlay, we have to rewind to undo */
				/* the read of this section */
};

/* Because V5, V6, V7 and 2.11BSD share several magic numbers
 * in their a.out headers, we must distinguish them so as to
 * set up the correct emulated environment. This is done by
 * observing the differences in their crt0.s code: they all
 * differ at position 021
 */
#define a_magic2	ov_siz[0]
#define V6_M2		0010600
#define V7_M2		0016600
#define BSD_M2		0162706


/* Array of 64K for data and instruction space */
static uint8_t darray[PDP_MEM_SIZE], iarray[PDP_MEM_SIZE];

#ifdef EMU211
/* 2.11BSD allows up to 16 8K overlays in the 0430 and 0431 a.out types.
 * Each overlay is loaded at the first 8K `click' above the end of the
 * main text. The following structures hold the overlays from the current
 * a.out, if there are any. Missing overlays have size 0 and pointer NULL.
 */
static struct {
    uint16_t size;
    uint8_t *ovlay;
} ovlist[NOVL]= {
    {0, NULL}, {0, NULL}, {0, NULL}, {0, NULL}, {0, NULL}, {0, NULL},
    {0, NULL}, {0, NULL}, {0, NULL}, {0, NULL}, {0, NULL}, {0, NULL},
    {0, NULL}, {0, NULL}, {0, NULL}
};

static uint8_t *ovbase;			/* Base address of 2.11BSD overlays */
#endif

/* Global array of pointers to arguments and environment. This
 * allows load_a_out() to modify it when dealing with shell
 * scripts, before calling set_arg_env()
 */
char *Argv[MAX_ARGS], *Envp[MAX_ARGS];
int Argc, Envc;

/* For programs without an environment, we set the environment statically.
 * Eventually there will be code to get some environment variables
 */
static char *default_envp[3] = {
	"PATH=/bin:/usr/bin:/usr/sbin:/usr/ucb:/usr/games:/usr/local/bin:.",
	"HOME=/",
	"TERM=vt100"
};
static int default_envc = 3;

static void set_arg_env(int v6exec);


/* Load the named PDP-11 executable file into the emulator's memory.
 * Returns 0 if ok, -1 if error. Also initialise the simulator and set
 * up the stack for the process with Argc, Argv, Envc, Envp.
 */
int
load_a_out(const char *file, int v6exec)
{
#define SCRIPT_LINESIZE 512		/* Max size of 1st line in script */
    FILE *zin;
    struct exec e;
    char bhash[2];
    char *script_line;
    char *script_arg[MAX_ARGS];
    int script_cnt=0;
    char **ap;
    uint8_t *ibase, *dbase, *bbase;	/* Instruction, data, bss bases */
    int size, i, j;
    int is_bsd=0;

    signal(SIGBUS, bus_error);	/* Catch all bus errors here */

				/* Open the file */
    zin = fopen(file, "r");
    if (zin == NULL) return (-1);

				/* First see if it is a script of some sort */
    if (fread(bhash, sizeof(bhash), 1, zin) != 1) {
	fclose(zin); return (-1);
    }
    if ((bhash[0]=='#') && (bhash[1]=='!')) {
				/* Get the first line of the file */
	if (((script_line=(char *)malloc(SCRIPT_LINESIZE))==NULL) ||
	    (fgets(script_line, SCRIPT_LINESIZE, zin)==NULL)) {
	    fprintf(stderr,"Apout - could not read 1st line of script\n");
	    fclose(zin); return(-1);
	}
				/* Now break into separate words */
        for (ap = script_arg; (*ap = strsep(&script_line, " \t\n")) != NULL;)
	    if (**ap != '\0') {
		ap++; script_cnt++;
	        if (script_cnt >= MAX_ARGS) break;
	    }
	fclose(zin);

#ifdef DEBUG
        if (trap_debug) {
	    fprintf(dbg_file, "Script: extra args are is %d\n", script_cnt);
	    for (i=0;i<script_cnt;i++)
	        fprintf(dbg_file," script_arg[%d] is %s\n",i,script_arg[i]);
	}
#endif

				/* Ensure we have room to shift the args */
	if ((Argc+script_cnt)>MAX_ARGS) {
	    fprintf(stderr,"Apout - out of argv space in script\n");
	    return(-1);
	}

				/* Now shift the args up and insert new ones */
	for (i=Argc-1; i!=0; i--)
	    Argv[i+script_cnt]= Argv[i];
	for (i=0; i<script_cnt; i++)
	    Argv[i]= script_arg[i];
	Argv[i]= strdup(file);
	Argc+= script_cnt;

	file= xlate_filename(script_arg[0]);
	return (load_a_out(file, v6exec));
    }				/* End of code for invoking scripts */


    
				/* Hmm, not a script, try to read */
				/* the a.out header */
    fseek(zin, 0, SEEK_SET);
    if (fread(&e, sizeof(e), 1, zin) != 1) {
	fclose(zin); return (-1);
    }
    if ((e.a_magic==BSD_OVERLAY) || (e.a_magic==BSD_ROVERLAY) || 
	(e.a_magic2==BSD_M2))
#ifndef EMU211
        { printf("Apout not compiled to support 2.11BSD binaries\n"); exit(1); }
#else
	is_bsd=1;
#endif
				/* Set up the memory areas according to */
				/* the magic numbers */
    switch (e.a_magic) {
    case ANY_NORMAL:
	fseek(zin, 16, SEEK_SET);	/* Move back to end of V5/6/7 header */
	ibase = ispace = dspace = darray;
	dbase = &(ispace[e.a_text]);
	bbase = &(ispace[e.a_text + e.a_data]);
	dwrite_base= e.a_text;
	break;
    case ANY_ROTEXT:
	fseek(zin, 16, SEEK_SET);	/* Move back to end of V5/6/7 header */
    case BSD_OVERLAY:
				/* Round up text area to next 8K boundary */
	if (e.a_text % EIGHT_K) {
	    size = EIGHT_K * (1 + e.a_text / EIGHT_K);
	} else {
	    size = e.a_text;
	}
				/* And the next 8K boundary if overlays! */
	if (e.a_magic==BSD_OVERLAY) {
	    if (e.max_ovl % EIGHT_K) {
	        size+= EIGHT_K * (1 + e.max_ovl / EIGHT_K);
	    } else {
	        size+= e.max_ovl;
	    }
	}
	ibase = ispace = dspace = darray;
	dbase = &(ispace[size]);
	bbase = &(ispace[size + e.a_data]);
	dwrite_base= size;
	break;
    case ANY_SPLITID:
	fseek(zin, 16, SEEK_SET);	/* Move back to end of V5/6/7 header */
    case BSD_ROVERLAY:
	ibase = ispace = iarray;
	dbase = dspace = darray;
	bbase = &(dspace[e.a_data]);
					/* Thinks: should we bother trying
					 * to stop programs doing this?
					 */
	if (is_bsd) {
	  dwrite_base=0;
	} else { dwrite_base=2; }	/* Try to stop null refs */
	break;
    case ANY_OVERLAY:
       fprintf(stderr,"Apout currently does not support the 0%o a.out format\n",
							ANY_OVERLAY);
	fclose(zin); return(-1);
    default:
	fprintf(stderr, "Apout - unknown a.out format 0%o\n", e.a_magic);
	fclose(zin); return(-1);
    }


    /* Initialise the instruction table for our environment */
#ifdef EMU211
    if (is_bsd)
	{ for (i=548; i<552; i++) itab[i]= bsdtrap; }
    else
#endif
	{ for (i=548; i<552; i++) itab[i]= v7trap; }

#ifdef ZERO_MEMORY
    memset(darray, 0, PDP_MEM_SIZE);	/* Clear all memory */
    if (ispace != dspace) memset(iarray, 0, PDP_MEM_SIZE);
#endif

    /* Now load the text into ibase */
    for (size = e.a_text; size;) {
	i = fread(ibase, 1, size, zin);
	if (i == -1) {
	    fclose(zin);
	    return (i);
	}
	size -= i;
	ibase += i;
    }

#ifdef EMU211
					/* Now deal with any overlays */
    if (is_bsd) switch (e.a_magic) {
    case BSD_OVERLAY:
    case BSD_ROVERLAY:
				/* Round up text area to next 8K boundary */
	if (e.a_text % EIGHT_K) {
	    size = EIGHT_K * (1 + e.a_text / EIGHT_K);
	} else {
	    size = e.a_text;
	}
	ovbase= &ispace[size];

	for (i=0;i<NOVL;i++) {
	    if (e.ov_siz[i]==0) {
		ovlist[i].size=0; ovlist[i].ovlay=NULL; continue;
	    }

			/* Create memory for the overlay */
	    ovlist[i].size=e.ov_siz[i];
	    if (ovlist[i].ovlay) free(ovlist[i].ovlay);
	    ovlist[i].ovlay= (uint8_t *)malloc(e.ov_siz[i]);
	    if (ovlist[i].ovlay==NULL) {
		fprintf(stderr,"Apout - can't malloc overlay!\n"); exit(1);
	    }
			/* Load the overlay into memory */
            for (size = ovlist[i].size, ibase=ovlist[i].ovlay; size;) {
	        j = fread(ibase, 1, size, zin);
		if (j == -1) {
	    	    fclose(zin);
	     	    return (j);
		}
	        size -= j;
	        ibase += j;
    	    }
	}

			/* And deal with the emt instructions */
	for (i=544; i<548; i++) itab[i]= do_bsd_overlay;

			/* Finally, load the first one */
        /* memcpy(ovbase, ovlist[0].ovlay, ovlist[0].size); */
    }
#endif

    /* Now load the data into dbase */
    if (dbase)
	for (size = e.a_data; size;) {
	    i = fread(dbase, 1, size, zin);
	    if (i == -1) {
		fclose(zin);
		return (i);
	    }
	    size -= i;
	    dbase += i;
	}

    /* Now clear the bss */
    if (bbase && e.a_bss)
	memset(bbase, 0, e.a_bss);


    regs[PC] = e.a_entry;
    fclose(zin);
    sim_init();
    set_arg_env(v6exec);
    return (0);
}

/*
 * C runtime startoff.	When an a.out is loaded by the kernel, the kernel
 * sets up the stack as follows:
 *
 *	_________________________________
 *	| (NULL)			| 0177776: top of memory
 *	|-------------------------------|
 *	|				|
 *	| environment strings		|
 *	|				|
 *	|-------------------------------|
 *	|				|
 *	| argument strings		|
 *	|				|
 *	|-------------------------------|
 *	| envv[envc] (NULL)		| end of environment vector tag, a 0
 *	|-------------------------------|
 *	| envv[envc-1]			| pointer to last environment string
 *	|-------------------------------|
 *	| ...				|
 *	|-------------------------------|
 *	| envv[0]			| pointer to first environment string
 *	|-------------------------------|
 *	| argv[argc] (NULL)		| end of argument vector tag, a 0
 *	|-------------------------------|
 *	| argv[argc-1]			| pointer to last argument string
 *	|-------------------------------|
 *	| ...				|
 *	|-------------------------------|
 *	| argv[0]			| pointer to first argument string
 *	|-------------------------------|
 * sp-> | argc				| number of arguments
 *	---------------------------------
 *
 * Crt0 simply moves the argc down two places in the stack, calculates the
 * the addresses of argv[0] and envv[0], putting those values into the two
 * spaces opened up to set the stack up as main expects to see it.
 * If v6exec is set, create a 5th/6th Edition environment. Otherwise,
 * create a 7th Edition or 2.11BSD environment.
 */
static void
set_arg_env(int v6exec)
{
    int i, posn, len;
    int eposn[MAX_ARGS];
    int aposn[MAX_ARGS];

			/* Set default environment if there is none */
    if (Envp[0]==NULL) {
	Envc=default_envc;
	for (i=0;i<Envc;i++) Envp[i]= default_envp[i];
    }

#ifdef DEBUG
    /* Set up the program's name -- used for debugging */
    if (progname) free(progname);
    progname = strdup(Argv[0]);

    if (trap_debug) {
	fprintf(dbg_file, "    In set_arg_env, Argc is %d\n", Argc);
	for (i=0;i<Argc;i++)
	    fprintf(dbg_file, "	Argv[%d] is %s\n", i, Argv[i]);
	for (i=0;i<Envc;i++)
	    fprintf(dbg_file, "	Envp[%d] is %s\n", i, Envp[i]);
    }
#endif

    /* Now build the arguments and pointers on the stack; see the Sixth */
    /* and Seventh Edition manuals for a description of the layout */
	
    posn = PDP_MEM_SIZE - 2;
    sl_word(posn, 0);		/* Put a NULL on top of stack */

    if (!v6exec)
	for (i = Envc - 1; i != -1; i--) {	/* For each env string */
	    len = strlen(Envp[i]) + 1;		/* get its length */
	    posn -= len;
	    memcpy(&dspace[posn], Envp[i], len);
	    eposn[i] = posn;
	}

    for (i = Argc - 1; i != -1; i--) {	/* For each arg string */
	len = strlen(Argv[i]) + 1;	/* get its length */
	posn -= len;
	memcpy(&dspace[posn], Argv[i], len);
	aposn[i] = posn;
    }
    posn -= 2;
    sl_word(posn, 0);			/* Put a NULL at end of env array */

    if (!v6exec) {			/* For each env string */
	for (i = Envc - 1; i != -1; i--) {
	    posn -= 2;
	    sl_word(posn, eposn[i]);	/* put a pointer to the string */
	}
	posn -= 2;
    }
				/* Put a NULL or -1 before arg pointers */
    if (v6exec) sl_word(posn, -1)
    else sl_word(posn, 0);

    for (i = Argc - 1; i != -1; i--) {	/* For each arg string */
	posn -= 2;
	sl_word(posn, aposn[i]);	/* put a pointer to the string */
    }
    posn -= 2;
    sl_word(posn, Argc);		/* Save the count of args */
    regs[SP] = posn;			/* and initialise the SP */
}


#ifdef EMU211
/* This function probably belongs in bsdtrap.c, but all the vars are
 * here, so why not!
 *
 * Deal with overlay changes which come in via an emt instruction.
 */

void do_bsd_overlay() {
    int ov= regs[0] - 1;

    if (ovlist[ov].size==0) {
	fprintf(stderr,"Apout- can't switch to empty overlay %d\n",ov);
    	exit(1);
    }

#ifdef DEBUG
    if (jsr_debug) fprintf(dbg_file, "switching to overlay %d\n", ov);
#endif

				/* Memcpy overlay into main ispace */
    memcpy(ovbase, ovlist[ov].ovlay, ovlist[ov].size);
}
#endif
