/* Copyright (c)1994-1999 Begemot Computer Associates. All rights reserved.
 * See the file COPYRIGHT for details of redistribution and use. */

/*
 * generate .s, .c or .o 'microcode dispatch table' from instruction list
 *
 * Usage is:
 *  geni [-mpv] [-o file] instruction-table cookie
 *
 * where:
 *	-m and
 *	-p	generate output for the cprof profiler
 *	-v	verbose
 *	-o file	outputs into file instead stdout and MUST be given for
 *		the object format outputs
 *
 * The instruction table format is described in ../ins.tab.
 * cookie is a string that says wich output to generate. See array
 * cookies[] for variants.
 *
 * Object format output is generated with the gnu bfd library. I tested
 * it with library version 2.5 (look into bfd.h). You also need -liberty
 * for the obstack package. Define HAVE_BFD to get object formats.
 */
# include <stdio.h>
# include <stdlib.h>
# include <stdarg.h>
# include <string.h>
# include <unistd.h>
# include <errno.h>

# include <begemot.h>
# include "cdefs.h"
# include "util.h"

# define HASHSIZE	10007
# define HASHJUMP	17

int	h_insert(u_char *);

void	fill(int, char *);
void	trans(int, int, char **);
void	appfunc(char *);
char	*strerror(int);

char	*hashtab[HASHSIZE];	/* name table */
int	hashsize;		/* number of names in table */

int 	code;			/* current instruction code */
int	ccc;			/* current microinstruction count */
int	coo = -1;		/* what output to generate */
int	profiler_output;
char	*ul;			/* the undeline character, if needed */
char	*ofile;			/* output file name */
int	verbose;
u_int	xtab[0x10000][4];	/* these point into the hash table */
char	*ifile;			/* file name */

# define COO_i386as	0
# define COO_self_as	1
# define COO_sun_as	2
# define COO_c		3
# define COO_i386_aout	0104
# define COO_obj	0100	/* must be ored in all non-text cookies */

struct {
	char	*coo;
	int	code;
} cookies[] = {
	{ "i386-as",		COO_i386as },
	{ "sparc-elf-as",	COO_self_as },
	{ "sparc-sunos-as",	COO_sun_as },
	{ "c",			COO_c },
# ifdef HAVE_LIBBFD
	{ "i386-aout",		COO_i386_aout },
# endif
};

void	do0(int, char **, int);
void	do1(int, char **, int);
void	doQ(int, char **, int);
void	doA(int, char **, int);
void	doS(int, char **, int);
void	doJ(int, char **, int);
void	doK(int, char **, int);
void	doD(int, char **, int);
void	doR(int, char **, int);
void	doF(int, char **, int);
void	doG(int, char **, int);
void	doH(int, char **, int);
void	doI(int, char **, int);
void	doB(int, char **, int);
int	define(char *);

void	out_c_defs();
void	do_profiler_output();
void	profiler_out_names();

void	getul();
int	ncmp(const void *, const void *);

void	obj_init_output();
void	obj_output_symtab();
void	obj_copy_temp();
void	obj_close();

void	open_output();
void	tab_out_i386as();
void	tab_out_self_as();
void	tab_out_sun_as();
void	tab_out_profiler();
void	tab_out_c();


int
main(int argc, char *argv[])
{
	char	buf[1000];
	char	*fields[20];
	uint	i, len;
	char	*end;
	int	opt;
	int	base, f;


	set_argv0(argv[0]);
	while((opt = getopt(argc, argv, "vmpo:")) != EOF)
		switch(opt) {

		case 'v':
			verbose++;
			break;

		case 'm':
			profiler_output = 1;
			break;
		case 'p':
			profiler_output = 2;
			break;

		case 'o':
			if((ofile = optarg) == 0)
				exit(2);
			break;
	}
	argc -= optind;
	argv += optind;

	if(argc != 2)
		panic("Usage: geni [-mvp] [-o file] file cookie\n");

	ifile = argv[0];
	if(strcmp(argv[0], "-")) {
		if(!freopen(argv[0], "r", stdin))
			panic("%s: %s\n", argv[0], strerror(errno));
		ifile = "stdin";
	}


	for(i = 0; i < ArraySize(cookies); i++)
		if(!strcmp(argv[1], cookies[i].coo)) {
			coo = cookies[i].code;
			break;
		}
	if(coo < 0)
		panic("unknown cookie '%s'\n", argv[1]);

	if((coo & COO_obj) && profiler_output)
		panic("no profiler options with .o output");

	getul();

	while(fgets(buf, sizeof(buf), stdin)) {
		if((len = strlen(buf)) > 0 && buf[len-1] == '\n')
			buf[len-1] = 0;
		f = getmfields(buf, fields, 20);
		if(f == 0 || fields[0][0] == '#')
			continue;
		if(f < 3)
			panic("missing fields: %s\n", buf);
		base = strtol(fields[0], &end, 8);
		if(*end)
			panic("syntax: %s\n", buf);
		if(base < code)
			panic("cannot go backward to %o\n", base);

		if(verbose)
			fprintf(stderr, "%s\n", fields[2]);

		fill(base, "ILL");

		switch(fields[1][0]) {
		case '0':	do0(base, fields+2, f-2); break;
		case '1':	do1(base, fields+2, f-2); break;
		case 'G':	doG(base, fields+2, f-2); break;
		case 'Q':	doQ(base, fields+2, f-2); break;
		case 'A':	doA(base, fields+2, f-2); break;
		case 'S':	doS(base, fields+2, f-2); break;
		case 'J':	doJ(base, fields+2, f-2); break;
		case 'K':	doK(base, fields+2, f-2); break;
		case 'D':	doD(base, fields+2, f-2); break;
		case 'R':	doR(base, fields+2, f-2); break;
		case 'F':	doF(base, fields+2, f-2); break;
		case 'H':	doH(base, fields+2, f-2); break;
		case 'I':	doI(base, fields+2, f-2); break;
		case 'B':	doB(base, fields+2, f-2); break;
		default: panic("bad type key: %s\n", buf);
		}
	}
	fill(0200000, "ILL");
	define("ILL");
	define("FILL");
	define("ret");

	if(profiler_output)
		profiler_out_names();

	switch(coo) {
	
	case COO_i386as:
		open_output();
		tab_out_i386as();
		break;

	case COO_self_as:
		open_output();
		tab_out_self_as();
		break;

	case COO_sun_as:
		open_output();
		tab_out_sun_as();
		break;

	case COO_c:
		open_output();
		if(profiler_output)
			tab_out_profiler();
		else
			tab_out_c();
		break;

	case COO_i386_aout:
		obj_init_output();
		obj_output_symtab();
		obj_copy_temp();
		obj_close();
		break;

	default:
		BAD_SWITCH(coo);
	}

	if(profiler_output == 2)
		do_profiler_output();

	return 0;
}

/*
 * if output file was given reopen stdout
 */
void
open_output()
{
	if(ofile == NULL || strcmp(ofile, "-") == 0)
		return;
	if(freopen(ofile, "w", stdout) == NULL)
		panic("%s: %s", ofile, strerror(errno));
}

/*
 * Output i386 gas format
 */
void
tab_out_i386as()
{
	printf("\t.file\t\"%s\"\n", ifile);
	printf("\t.globl\t_instab\n");
	printf(".text\n");
	printf("\t.align\t2\n");
	printf("_instab:\n");
	for(code = 0; code < 0x10000; code++) {
		printf("\t.long\t");
		for(ccc = 0; ccc < 4; ccc++)
			printf("%s%s", hashtab[xtab[code][ccc]], (ccc < 3) ? ", " : "");
		printf("\t# %06o\n", code);
	}

}

/*
 * output SunOs Assembler format
 */
void
tab_out_sun_as()
{
	printf("\t.global\t_instab\n");
	printf(".text\n");
	printf("\t.align\t2\n");
	printf("_instab:\n");
	for(code = 0; code < 0x10000; code++) {
		printf("\t.long\t");
		for(ccc = 0; ccc < 4; ccc++)
			printf("%s%s", hashtab[xtab[code][ccc]], (ccc < 3) ? ", " : "");
		printf("\t! %06o\n", code);
	}
}

/*
 * output sparc elf format
 */
void
tab_out_self_as()
{
	printf("\t.global\tinstab\n");
	printf(".section\t\".text\"\n");
	printf("\t.align\t4\n");
	printf("\t.type\tinstab,#object\n");
	printf("\t.size\tinstab,%lu\n", (1L << 16) * 4 * 4);
	printf("instab:\n");
	for(code = 0; code < 0x10000; code++) {
		printf("\t.word\t");
		for(ccc = 0; ccc < 4; ccc++)
			printf("%s%s", hashtab[xtab[code][ccc]], (ccc < 3) ? ", " : "");
		printf("\n");
	}
}

/*
 * output C-code
 */
void
tab_out_c()
{
	out_c_defs();
	printf("void (*const instab[0200000][4])(void) = {\n");
	for(code = 0; code < 0x10000; code++) {
		printf("/* %06o */ { ", code);
		for(ccc = 0; ccc < 4; ccc++)
			printf("%s%s", hashtab[xtab[code][ccc]], (ccc < 3) ? ", " : "");
		printf(" },\n");
	}
	printf("};\n");
}

/*
 * output C-code for profiler
 */
void
tab_out_profiler()
{
	printf("int instab[0200000][4] = {\n");
	for(code = 0; code < 0x10000; code++) {
		printf("/* %06o */ { ", code);
		for(ccc = 0; ccc < 4; ccc++)
			printf("%s%s", hashtab[xtab[code][ccc]], (ccc < 3) ? ", " : "");
		printf(" },\n");
	}
	printf("};\n");
}


/*
 * print function prototypes for all symbols
 */
void
out_c_defs()
{
	int	i;

	for(i = 0; i < HASHSIZE; i++)
		if(hashtab[i] != 0)
			printf("void %s(void);\n", hashtab[i]);
}

/*
 * start a new instruction code
 */
void
start(void)
{
	ccc = 0;
}


/*
 * done with this instruction code, fill microcode slots with returns
 */
void
theend(void)
{
	while(ccc < 4)
		appfunc("ret");
	code++;
}

/*
 * fill instruction code with constant micro-function
 */
void	
fill(int to, char *w)
{
	while(code < to) {
		start();
		appfunc(w);
		theend();
	}
}

/*
 * one code only
 */
void	do0(int base UNUSED, char **fields, int f UNUSED)
{
	start();
	appfunc(fields[0]);
	define(fields[0]);
	theend();
}

/*
 * dito, but do translation
 */
void	do1(int base UNUSED, char **fields, int f)
{
	start();
	trans(0, f, fields);
	theend();
}

/* 
 * JMP - register address is illegal
 */
void	doG(int base, char **fields, int f)
{
	while(code < base + 0100) {
		start();
		if((code & 070) == 0)
			appfunc("ILL");
		else
			trans(code, f, fields);
		theend();
	}
}

/*
 * RTS/SPL last 3 bits are arg
 */
void	doQ(int base, char **fields, int f UNUSED)
{
	while(code < base + 010) {
		start();
		appfunc(fields[0]);
		define(fields[0]);
		theend();
	}
}

/*
 * SCC/CCC last 4 bits are arg
 */
void	doA(int base, char **fields, int f UNUSED)
{
	while(code < base + 020) {
		start();
		appfunc(fields[0]);
		define(fields[0]);
		theend();
	}
}

/*
 * single operand
 */
void	doS(int base, char **fields, int f)
{
	while(code < base + 0100) {
		start();
		trans(code, f, fields);
		theend();
	}
}

/*
 * single operand but no mode 0
 */
void	doB(int base, char **fields, int f)
{
	while(code < base + 0100) {
		start();
		if((code & 070) == 0)
			appfunc("ILL");
		else
			trans(code, f, fields);
		theend();
	}
}

/*
 * BRs, TRAP, EMT lower byte is arg
 */
void	doJ(int base, char **fields, int f UNUSED)
{
	while(code < base + 0400) {
		start();
		appfunc(fields[0]);
		define(fields[0]);
		theend();
	}
}

/*
 * JSR 
 */
void	doK(int base, char **fields, int f)
{
	while(code < base + 01000) {
		start();
		if((code & 070) == 0)
			appfunc("ILL");
		else
			trans(code, f, fields);
		theend();
	}
}

/*
 * double ops
 */
void	doD(int base, char **fields, int f)
{
	while(code < base + 010000) {
		start();
		trans(code, f, fields);
		theend();
	}
}

/*
 * double register and source/dest op
 */
void	doR(int base, char **fields, int f)
{
	while(code < base + 01000) {
		start();
		trans(code, f, fields);
		theend();
	}
}

/* 
 * FPP with single float destination
 * mode 0 with register 6 and 7 is illegal
 */
void	doF(int base, char **fields, int f)
{
	while(code < base + 0100) {
		start();
		if((code & 077) == 006 || (code & 077) == 007)
			appfunc("FILL");
		else
			trans(code, f, fields);
		theend();
	}
}

/*
 * FPP - one operand is integer source the other FPP reg.
 */
void	doH(int base, char **fields, int f)
{
	while(code < base + 0400) {
		start();
		trans(code, f, fields);
		theend();
	}
}

/*
 * FPP - two float ops
 * mode 06 and 07 on source is illegal
 */
void	doI(int base, char **fields, int f)
{
	while(code < base + 0400) {
		start();
		if((code & 077) == 006 || (code & 077) == 007)
			appfunc("FILL");
		else
			trans(code, f, fields);
		theend();
	}
}

void	trans(int v, int f, char **fields)
{
	int i;
	char	nbuf[100];

	for(i = 1; i < f; i++) {
		if(fields[i][0] == '%') {
			appfunc(fields[0]);
			define(fields[0]);
		} else switch(fields[i][1]) {
		case '0':	/* DD */
			sprintf(nbuf, "%c%02o", fields[i][0], v & 077);
			if( profiler_output != 2 )
				appfunc(nbuf);
			define(nbuf);
			break;
		case '1':	/* SS */
			sprintf(nbuf, "%c%02o", fields[i][0], (v >> 6) & 077);
			if( profiler_output != 2 )
				appfunc(nbuf);
			define(nbuf);
			break;
		case '2':	/* FPP */
			sprintf(nbuf, "FPP_%c", fields[i][0]);
			if( profiler_output != 2 )
				appfunc(nbuf);
			define(nbuf);
			break;
		default:
			panic("bad translation key: %c\n", fields[i][1]);
		}
	}
}

void
getul()
{
	switch(coo) {

	case COO_i386as:
	case COO_sun_as:
	case COO_i386_aout:
		ul = "_";
		break;

	case COO_self_as:
	case COO_c:
		ul = "";
		break;

	default:
		BAD_SWITCH(coo);
	}
}

void
appfunc(char *s)
{
	xtab[code][ccc++] = define(s);
}


/*
 * compute a hash code from a string
 * we simly add the unsigned chars here
 */
u_int
h_code(u_char *s)
{
	u_int	h = 0;

	while(*s)
		h += *s++;
	return h;
}

int
h_insert(u_char *s)
{
	u_int	h0 = h_code(s);
	u_int	h = h0;

	do {
		if(hashtab[h] == NULL) {
			if((hashtab[h] = strcpy(malloc(strlen(s)+1), s)) == 0)
				return -1;
			hashsize++;
			return h;
		}
		if(strcmp(hashtab[h], s) == 0)
			return h;
		h = (h + HASHJUMP) % HASHSIZE;
	} while(h != h0);
	return -2;
}

int
define(char *n)
{
	int	 i;
	char	buf[100];

	strcpy(buf, ul);
	strcat(buf, n);

	if((i = h_insert(buf)) >= 0)
		return i;

	if(i == -1)
		panic("out of memory");
	if(i == -2)
		panic("recompile me with greater HASHSIZE!");
	panic("this is unpossible :-/");
}

void
do_profiler_output()
{
	FILE	*fp;
	int	i, j;

	if((fp = fopen("cproftab.c", "w")) == NULL)
		panic("cproftab.c: %s", strerror(errno));

	fprintf(fp, "char *inames[] = {\n");
	for(i = j = 0; i < HASHSIZE; i++)
		if(hashtab[i] != 0)
			fprintf(fp, "\t\"%s\",	/* %3d */\n", hashtab[i], j++);
	fprintf(fp, "};\n\n");

	fprintf(fp, "# define NNAMES %d\n", hashsize);

	fclose(fp);
}


/*
 * print sorted defines for the names
 */
void
profiler_out_names()
{
	char	**ntab;
	int	i, j;

	ntab = xalloc(sizeof(char *) * hashsize);
	for(j = i = 0; i < HASHSIZE; i++)
		if(hashtab[i] != 0)
			ntab[j++] = hashtab[i];

	qsort(ntab, hashsize, sizeof(char *), ncmp);
	for(i = 0; i < hashsize; i++)
		printf("# define %s %d\n", ntab[i], i);

	free(ntab);
}


int
ncmp(const void *p1, const void *p2)
{
	return strcmp(*(char **)p1, *(char **)p2);
}


# ifdef HAVE_LIBBFD

# include <bfd.h>

# define ENTRIES	(0x10000 * 4)
# define PTRSIZE	(sizeof(void (*)(void)))
# define TEXTSIZE	(ENTRIES * PTRSIZE)

PTR bfd_alloc_by_size_t(bfd *, size_t);

bfd		*abfd;		/* the output file */
asection	*text_sec;	/* .text section */
asymbol		**symbols;	/* symbol pointers */
int		nsymbols;	/* number of symbols */
arelent		**relocs;	/* relocations */
int		nrelocs;	/* number of relocations */
int		symptr[HASHSIZE];

/*
 * initialize bfd stuff
 *	create bfd
 *	create .text section
 *	code assumes that void (*)(void) are the same size as on target
 */
void
obj_init_output()
{
	char	*fmt;

	switch(coo) {

	case COO_i386_aout:
		fmt = "a.out-i386-bsd";
		break;

	default:
		BAD_SWITCH(coo);
	}

	if(ofile == 0)
		panic("need -o for object output");

	if((abfd = bfd_openw(ofile, fmt)) == 0)
		panic("bfd_openw(%s): %s", ofile, bfd_errmsg(bfd_get_error()));
	if(!bfd_set_format(abfd, bfd_object))
		panic("bfd_set_format: %s", bfd_errmsg(bfd_get_error()));

	if((text_sec = bfd_make_section_old_way(abfd, ".text")) == 0)
		panic("bfd_make_section_old_way: %s", bfd_errmsg(bfd_get_error()));
	if(!bfd_set_section_size(abfd, text_sec, TEXTSIZE))
		panic("bfd_set_section_size: %s", bfd_errmsg(bfd_get_error()));
}


/*
 * create symbol table from name table
 */
void
obj_output_symtab()
{
	asymbol	*new;
	int	i;

	if((symbols = (asymbol **)bfd_alloc_by_size_t(abfd, (hashsize + 2) * sizeof(asymbol *))) == 0)
		panic("out of memory");

	if((new = bfd_make_empty_symbol(abfd)) == 0)
		panic("bfd_make_empty_symbol: %s", bfd_errmsg(bfd_get_error()));

	new->name = "_instab";
	new->section = text_sec;
	new->flags = BSF_GLOBAL;
	new->value = 0x0;

	symbols[nsymbols++] = new;

	for(i = 0; i < HASHSIZE; i++) {
		if(hashtab[i] != 0) {
			if((new = bfd_make_empty_symbol(abfd)) == 0)
				panic("bfd_make_empty_symbol: %s", bfd_errmsg(bfd_get_error()));

			new->section = bfd_und_section_ptr;
			new->value = 0;
			new->name = hashtab[i];

			symbols[nsymbols] = new;
			symptr[i] = nsymbols++;
		}
	}
}


/*
 * read temporary file and create relocations
 */
void
obj_copy_temp()
{
	u_long	off = 0;
	arelent	*rel;
	u_int	null = 0;
	char	*err;
	arelent	*rtab;

	if((relocs = (arelent **)bfd_alloc_by_size_t(abfd, (ENTRIES + 1) * sizeof(arelent *))) == 0)
		panic("out of memory");
	if((rtab = (arelent *)bfd_alloc_by_size_t(abfd, ENTRIES * sizeof(arelent))) == 0)
		panic("out of memory");

	for(code = 0; code < 0x10000; code++) {
		if(verbose && code % 0x1000 == 0)
			fprintf(stderr, "code %lx\r", off);

		for(ccc = 0; ccc < 4; ccc++) {
			rel = &rtab[nrelocs];
			rel->sym_ptr_ptr = &symbols[symptr[xtab[code][ccc]]];
			rel->address = off;
			rel->addend = 0;
			rel->howto = bfd_reloc_type_lookup(abfd, BFD_RELOC_32);

			if(bfd_install_relocation(abfd, rel, &null, 0, text_sec, &err) != bfd_reloc_ok)
				panic("bfd_install_relocation: %s", err);

			relocs[nrelocs++] = rel;

			off += PTRSIZE;
		}
	}
}


/*
 * set symbol table and relocations and close file
 */
void
obj_close()
{
	symbols[nsymbols] = 0;
	relocs[nrelocs] = 0;

	if(!bfd_set_symtab(abfd, symbols, nsymbols))
		panic("bfd_set_symtab: %s", bfd_errmsg(bfd_get_error()));
	bfd_set_reloc(abfd, text_sec, relocs, nrelocs);

	if(!bfd_close(abfd))
		panic("bfd_close: %s", bfd_errmsg(bfd_get_error()));
}

# else
void obj_init_output() { panic("gulp?"); }
void obj_output_symtab() { panic("gulp?"); }
void obj_copy_temp() { panic("gulp?"); }
void obj_close() { panic("gulp?"); }
# endif
