/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

/*
 * sparc floating point operations for compilers that don't know extended float
 * implemented in as because gcc doesn't know about extended doubles
 *
 * See the SPARC Architecture Manual, Version 8, page 238, for further
 * information on Quad Precision Floating Point.
 */

/*
 * sparc quad float format, usually emulated
 */
struct Float {
	unsigned	s : 1;		/* sign bit */
	unsigned	e : 15;		/* expo offset 0x3fff */
	unsigned	m0 : 16;	/* mantissa 112 bit */
	unsigned	m1 : 32;
	unsigned	m2 : 32;
	unsigned	m3 : 32;
};
# define EXP_OFF	0x3fff

# define InitFp()
# define EndFp()

/* 
 * add S to A
 * sub S from A
 * mul A by S
 * divide A by S (S is already checked for 0)
 * the limits of the host float should be sufficient not
 * to produce any errors.
 */
# ifdef HW_QUAD
# define AddF(A,S)	asm("	ldd	[%1], %%f0		;\
				ldd	[%1 + 0x8], %%f2	;\
				ldd	[%0], %%f4		;\
				ldd	[%0 + 0x8], %%f6	;\
				faddq	%%f0, %%f4, %%f4	;\
				std	%%f6, [%0 + 0x8]	;\
				std	%%f4, [%0]		\
				" : : "r"(&A), "r"(&S) : "f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7")

# define SubF(A,S)	asm("	ldd	[%1], %%f0		;\
				ldd	[%1 + 0x8], %%f2	;\
				ldd	[%0], %%f4		;\
				ldd	[%0 + 0x8], %%f6	;\
				fsubq	%%f4, %%f0, %%f4	;\
				std	%%f6, [%0 + 0x8]	;\
				std	%%f4, [%0]		\
				" : : "r"(&A), "r"(&S) : "f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7")
# define MulF(A,S)	asm("	ldd	[%1], %%f0		;\
				ldd	[%1 + 0x8], %%f2	;\
				ldd	[%0], %%f4		;\
				ldd	[%0 + 0x8], %%f6	;\
				fmulq	%%f4, %%f0, %%f4	;\
				std	%%f6, [%0 + 0x8]	;\
				std	%%f4, [%0]		\
				" : : "r"(&A), "r"(&S) : "f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7")
# define DivF(A,S)	asm("	ldd	[%1], %%f0		;\
				ldd	[%1 + 0x8], %%f2	;\
				ldd	[%0], %%f4		;\
				ldd	[%0 + 0x8], %%f6	;\
				fdivq	%%f4, %%f0, %%f4	;\
				std	%%f6, [%0 + 0x8]	;\
				std	%%f4, [%0]		\
				" : : "r"(&A), "r"(&S) : "f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7")

/*
 * separate A in fractional part F and integral part I
 * this version breaks on big floats
 */
# define ModF(A,F,I)	asm("	ldd	[%0], %%f0		;\
				ldd	[%0+0x8], %%f2		;\
				fqtoi	%%f0, %%f4		;\
				fitoq	%%f4, %%f4		;\
				std	%%f6, [%2 + 0x8]	;\
				std	%%f4, [%2]		;\
				fsubq	%%f0, %%f4, %%f4	;\
				std	%%f6, [%1 + 0x8]	;\
				std	%%f4, [%1]		\
				" :: "r"(&A), "r"(&F),  "r"(&I) : "l0", "f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7")

/*
 * check if S is zero
 */
# define IsZ(S)		({ int _r;				;\
			asm("	ldd	[%1], %%f0		;\
				ldd	[%1+0x8], %%f2		;\
				ldd	[%2], %%f4		;\
				ldd	[%2+0x8], %%f6		;\
				fcmpq	%%f0, %%f4		;\
				nop				;\
				fbe,a	1f			;\
				mov	1,%0			;\
				mov	0,%0			;\
			1:					;\
				" : "=r"(_r) : "r"(&S), "r"(&host_null) : "f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7"); \
			_r; })

/*
 * check if S is negativ
 */
# define IsN(S)			((S).s)


/*
 * convert long to float
 */
# define CnvL(A,L)	asm("	ld	[%1], %%f0		;\
				fitoq	%%f0, %%f0		;\
				std	%%f2, [%0 + 0x8]	;\
				std	%%f0, [%0]		\
				" : : "r"(&A), "r"(&L) : "f0", "f1", "f2", "f3")

/*
 * chop float to long, range already checked
 * chop already set by InitFp
 */
# define CnvF2L(S,L)	asm("	ldd	[%1], %%f0		;\
				ldd	[%1+0x8], %%f2		;\
				fqtoi	%%f0, %0		\
				" : "=f"(L) : "r"(&S) : "f0", "f1", "f2", "f3")

# else HW_QUAD

struct Float	_Q_add(struct Float, struct Float);
struct Float	_Q_sub(struct Float, struct Float);
struct Float	_Q_mul(struct Float, struct Float);
struct Float	_Q_div(struct Float, struct Float);
int		_Q_qtoi(struct Float);
double		_Q_qtod(struct Float);
struct Float	_Q_itoq(int);
int		_Q_feq(struct Float, struct Float);
int		_Q_fgt(struct Float, struct Float);
int		_Q_fge(struct Float, struct Float);

struct Float 	modq(struct Float, struct Float *);

# define AddF(A,S)	(A) = _Q_add((A), (S))
# define SubF(A,S)	(A) = _Q_sub((A), (S))
# define MulF(A,S)	(A) = _Q_mul((A), (S))
# define DivF(A,S)	(A) = _Q_div((A), (S))
# define ModF(A,F,I)	(F) = modq((A), &(I))
# define IsZ(A)		_Q_feq((A), host_null)
# define IsN(S)		((S).s)
# define CnvL(A,L)	(A) = _Q_itoq((L))
# define CnvF2L(S,L)	(L) = _Q_qtoi((S))

# endif HW_QUAD


/*
 * return K so that 2^K <= A < 2^(K+1)
 * (A may not be 0 here)
 */
# define FrExp(A)	((A).e - EXP_OFF)


/*
 * return sparc float mantissa as a 64bit fixed point value with
 * point after the 1st bit to the left and 1.0 > m >= 0.5
 */
# define GetMant(P)	(((UQuad)(P)->m0 << 47) |		\
			 ((UQuad)(P)->m1 << 15) |		\
			 ((UQuad)(P)->m2 >> 17))
# define GetExp(P)	((int)(sshort)(P)->e - EXP_OFF + 1)
# define GetSign(P)	(P)->s


/*
 * insert mantissa into sparc float
 */
# define SetMant(H,M)	(((H)->m0 = (M) >> 47), ((H)->m1 = (M) >> 15),	\
			 ((H)->m2 = (M) << 17), ((H)->m3 = 0))
# define SetExp(H,E)	((H)->e = (E) - 1 + EXP_OFF)
# define SetSign(H,S)	((H)->s = (S))


/*
 * covert extended float to double, just for printing and debuging
 */
static double
mkd(Float *f)
{
# ifdef HW_QUAD
	static double	dd;

	asm("   ldd     [%1], %%f0              ;\
		ldd     [%1 + 0x8], %%f2        ;\
		fqtod	%%f0, %0"
	  : "=f"(dd) : "r"(f) : "f0", "f1", "f2", "f3");
	return dd;
# else
	return _Q_qtod(*f);
# endif
}


/*
 * dump a float
 */
static void
dmpflt(Float *f)
{
	int e = f->e - EXP_OFF;

	printf("%c ", "+-"[f->s]);
	if(f->e)
		printf(" %04x ", e & 0x7fff);
	else 
		printf("    0 ");	
	printf("%04x %08x %08x %08x", f->m0, f->m1, f->m2, f->m3);
}


/*
 * convert pdp float to double for printing
 * precision may be lost and over/underflow may occure
 */
static double
mkdouble(PDP_Float *f, int d)
{
	Float h;

	unfold(f, &h, d);
	return mkd(&h);
}


Float
modq(Float val, Float *iptr)
{
	static Float maxp2 = { 0, 112+0x3FFF, 0, 0, 0, 0 };
	static Float c1 = { 0, 0x3FFF, 0, 0, 0, 0 };
	Float aval = val;

	aval.s = 0;
	if(_Q_fge(aval, maxp2))
		*iptr = val;
	else {
		*iptr = _Q_add(aval, maxp2);
		*iptr = _Q_sub(*iptr, maxp2);
		while(_Q_fgt(*iptr, aval))
			*iptr = _Q_sub(*iptr, c1);
		if(val.s)
			iptr->s = 1;
	}
	return _Q_sub(val, *iptr);
}
