#
/*
 | RX11 floppy disk driver
 |
 | Implements 3 kinds of UNIX block devices;
 | Minor Device assignment:
 |	<0, 1>	Simple mapping of block numbers into diskette
 |		sectors, starting at track 1 (for proposed ANSII
 |		compatabilty).
 |		Each block (0-493) occupies 4 consecutive sectors.
 |
 |	<2, 3>	RT-11 compatible sector interleave - 2:1 with
 |		6 sector skew across tracks.
 |		This mapping also leaves track 0 out of reach.
 |
 |	<4, 5>	Simple sector mapping, starting at track 0, sector 1.
 |		This leaves the last 2 sectors on the last track (66)
 |		unused.
 */
/* if the line:
#define RXSWP	RXSWP
is included, the unix swap device can be a floppy disk...
*/

#include "../param.h"
#include "../buf.h"
#include "../conf.h"
#include "../user.h"

#define RXADDR	0177170
#define RX_MAXB	494	/* blocks per RX diskette (devices 0, 1) */
#define RX_MAXB1 500	/* blocks in devices 4, 5 */

/* Function Codes */
#define RX_FILL		000 /* fill buffer (before a write) */
#define RX_EMPTY	002 /* empty buffer (after a read) */
#define RX_WRITE	004 /* write sector */
#define RX_READ		006 /* read sector */
#define RX_RDSTAT	012 /* read status register */
#define RX_WDELD	014 /* write deleted data sector */
#define RX_RDERR	016 /* read error register */

/* Command & Status Bits */
#define GO	01	/* initiate a command */
#define DRV1	020	/* selects drive 1 when set, else 0 */
#define DONE	040	/* indicates function completion */
#define IENABLE	0100	/* interrupt enable on DONE */
#define TR	0200	/* transfer request */
#define RXINIT	040000	/* initialize the RX11 */
#define RXERR	0100000	/* error bit */

/* internal flags (in rxtab.d_active) */
#define X_BUSY	01	/* am doing somthing */
#define X_SOFT	02	/* soft error - retry */
#define X_HARD	04	/* hard error - abort */
#define X_DONE	06	/* function completed */
#define X_SLP	010	/* sleep on seeks */
#define X_INIT	020	/* doing RXINIT */

struct {
	int rxcs;
	int rxdb;
};
struct devtab rxtab;
int rx_oflag;		/* open flags for each drive */

rxopen(dev)
{
	register mdev;
	register unit;

	mdev = dev.d_minor;
	unit = 1<<(mdev&01);
	if(mdev > 5 || rx_oflag&unit) {
		u.u_error = ENXIO;
		return;
	}
	if(rxtab.d_active&X_INIT)
		sleep(&rxtab, PRIBIO);
	if(rx_oflag == 0 && RXADDR->rxcs < 0) {
		u.u_error = EIO;
		return;
	}
	rx_oflag =| unit;
}

rxclose(dev)
{
	register mdev;
	register unit;
	register struct buf *bp;
	extern wakeup;

	mdev = dev.d_minor;
	unit = 1<<(mdev&01);
	if(mdev > 5 || (rx_oflag&unit) == 0) {
		u.u_error = ENXIO;
		return;
	}
	for(bp = rxtab.b_forw; bp != &rxtab; bp = bp->b_forw)
		if(bp->b_dev.d_minor == mdev) {
			bflush(bp);
			bp->b_dev = NODEV;
		}
	if(rx_oflag == unit) {
		rxtab.d_active =| X_INIT;
		RXADDR->rxcs = RXINIT;
		do {
			/* no interrupt on init done ! */
			timeout(wakeup, &rx_oflag, HZ);
			sleep(&rx_oflag, PRIBIO);
		} while((RXADDR->rxcs&DONE) == 0);
		rxtab.d_active = 0;
		wakeup(&rxtab);
	}
	rx_oflag =& (~unit);
}

rxstrategy(abp)
struct buf *abp;
{
	register struct devtab *d;
	register struct buf *bp;

	bp = abp;
	if((bp->b_dev.d_minor > 5) ||
		(bp->b_dev.d_minor < 4 && bp->b_blkno >= RX_MAXB) ||
		(bp->b_dev.d_minor > 3 && bp->b_blkno >= RX_MAXB1) ) {
			bp->b_flags =| B_ERROR;
			iodone(bp);
			return;
	}
	d = &rxtab;
	bp->av_forw = 0;
	spl5();
	if(d->d_actf == 0)
		d->d_actf = bp;
	else
		d->d_actl->av_forw = bp;
	d->d_actl = bp;
	if(d->d_active == 0)
		rxio();
	spl0();
}

rxintr()
{
	register struct devtab *d;
	register struct buf *bp;

	d = &rxtab;
	if(d->d_active == 0)
		return;
	bp = d->d_actf;
	if(RXADDR->rxcs < 0) {	/* error bit */
		if(++d->d_errcnt <= 5) {
			d->d_active =| X_SOFT;
			goto out;
		}
		deverror(bp, RXADDR->rxdb, 0);
		d->d_active =| X_HARD;
		bp->b_flags =| B_ERROR;
	} else
		d->d_active =| X_DONE;
	d->d_errcnt = 0;
out:
	if(d->d_active&X_SLP) {
		d->d_active =& (~X_SLP);
		wakeup(d);
	}
}

rxio()
{
	register struct buf *bp;
	register char *cp;
	register drive;
	int sect, trk, logsec;

start:
	if((bp=rxtab.d_actf) == 0)
		return;
	rxtab.d_active = X_BUSY;
	spl0();
#ifdef RXSWP
	if(bp->b_flags&B_PHYS) {
		rxswap(bp);
		goto out;
	}
#endif
	drive = (bp->b_dev.d_minor&01)? DRV1: 0;
	logsec = bp->b_blkno<<2;
	for(cp=bp->b_addr; cp < &(bp->b_addr[512]); cp =+ 128) {
		sect = rxaddr(logsec++, bp->b_dev.d_minor, &trk);
		if(bp->b_flags&B_READ) {
			if(!rxhome(trk, sect, drive, B_READ))
				break;
			if(!rxbuf(cp, B_READ))
				break;
		} else {
			if(!rxbuf(cp, 0))
				break;
			if(!rxhome(trk, sect, drive, 0))
				break;
		}
	}
out:
	spl6();		/* U-G-L-Y */
	rxtab.d_actf = bp->av_forw;
	rxtab.d_active = 0;
	iodone(bp);
	goto start;
}

/* Map logical sector numbers into physical
 | tracks and sectors.
 | The meaning of minor device numbers is hidden
 | right here in this code.
 */

#define NSECT	26	/* sectors per track */
#define SKEW	6	/* track to track skew */
rxaddr(logsec, adev, trakp)
char *trakp;	/* call by ref. */
{
	register trk, sect, mdev;

	mdev = adev;	/* minor device number */
	trk = logsec/NSECT;
	sect = logsec%NSECT;
	if(mdev < 2 || mdev > 3) {

		/* simple sector mapping, devices 0,1,4,5 */
		sect++;
		if(mdev < 2)
			trk++;	/* leave track 0 alone (dev 0,1) */
	} else {

		/* std. RT-11 sector interleave (devices 2,3) */
		sect = (sect<<1) | (sect >= (NSECT/2));
		sect =+ (SKEW*trk++);	/* don't touch track 0 */
		sect = sect%NSECT + 1;
	}
	*trakp = trk;		/* transfer back computed track address */
	return(sect);		/* and ... sector number */
}

/*
 | rxhome - Home on a sector;
 |	executes the "read sector" and "write sector" commands.
 */

rxhome(trk, sect, drive, rw)
{
	register struct devtab *d;
	register char *rxloc;
	register com;

	d = &rxtab;
	rxloc = RXADDR;
	com = (rw? RX_READ: RX_WRITE) | IENABLE | drive | GO;
	again:
		d->d_active = X_BUSY;
		rxloc->rxcs = com;
		while((rxloc->rxcs&(TR|DONE)) == 0);
		switch(d->d_active&X_DONE) {
		case X_SOFT:	goto again;
		case X_HARD:	return(0);
		}
		rxloc->rxdb = sect;
		while((rxloc->rxcs&(TR|DONE)) == 0);
		switch(d->d_active&X_DONE) {
		case X_SOFT:	goto again;
		case X_HARD:	return(0);
		}
		spl5();
		d->d_active =| X_SLP;
		rxloc->rxdb = trk;
		sleep(d, PRIBIO);
		spl0();
		switch(d->d_active&X_DONE) {
		case X_SOFT:	goto again;
		case X_HARD:	return(0);
		}
	return(1);
}

/*
 | Executes the "fill buffer" and "empty buffer" command
 | sequences.
 */
rxbuf(addr, rw)
{
	register char *p;
	register n, com;

	com = (rw? RX_EMPTY: RX_FILL) | IENABLE | GO;

	again:
		rxtab.d_active = X_BUSY;	/* clear errors */
		RXADDR->rxcs = com;
		n=128;	/* delay ?? */
		while((RXADDR->rxcs&(TR|DONE)) == 0);
		switch(rxtab.d_active&X_DONE) {
		case X_SOFT:	goto again;
		case X_HARD:	return(0);
		}
		p = addr;
		do {
			if(rw)
				*p++ = RXADDR->rxdb;
			else
				RXADDR->rxdb = *p++;
			while((RXADDR->rxcs&(TR|DONE)) == 0);
			switch(rxtab.d_active&X_DONE) {
			case X_SOFT:	goto again;
			case X_HARD:	return(0);
			}
		} while(--n);
	return(1);
}

#ifdef RXSWP		/* do floppy swap io */
int rxswbuf[64];	/* swap buffer */
rxswap(abp)
struct buf *abp;
{
	register struct buf *bp;
	register int rw;
	register char *phaddr;
	int logsec, drive, sect, trk;

	bp = abp;
	rw = bp->b_flags&B_READ;
	drive = (bp->b_dev.d_minor&1)? DRV1: 0;
	phaddr = ((bp->b_addr>>6)&01777) | ((bp->b_xmem&077)<<10);
	logsec = bp->b_blkno<<2;
	for(; bp->b_wcount < 0; bp->b_wcount =+ 32) {
		sect = rxaddr(logsec++, bp->b_dev.d_minor, &trk);
		if(rw) {
			if(!rxhome(trk, sect, drive, rw))
				break;
			if(!rxbuf(rxswbuf, rw))
				break;
			kuseg(rxswbuf, phaddr++, rw);
			if((bp->b_wcount =+ 32) < 0)
				kuseg(&rxswbuf[32], phaddr++, rw);
		} else {
			kuseg(rxswbuf, phaddr++, rw);
			if((bp->b_wcount =+ 32) < 0)
				kuseg(&rxswbuf[32], phaddr++, rw);
			if(!rxbuf(rxswbuf, rw))
				break;
			if(!rxhome(trk, sect, drive, rw))
				break;
		}
	}
}
#endif
