/* $Header: /usr/alex/drivers/sb16/RCS/sb16mixer.c,v 1.3 94/05/21 11:46:59 alex Exp $
 *
 * Sound Blaster 16 Mixer Interface
 *
 * Copyright (c) 1994 Alex Nash, All Rights Reserved.
 *
 * This software is provided "as is" and without warranty of any kind.
 * The author accepts no responsibility for any damage caused by this
 * software.
 *
 * $Log:	sb16mixer.c,v $
 * Revision 1.3  94/05/21  11:46:59  alex
 * Ability to configure input mixer for mono recordings.
 * 
 * Revision 1.2  94/04/12  19:18:00  alex
 * Added mute capability and support for read/write device permissions.
 *
 * Revision 1.1  94/04/07  17:19:34  alex
 * Initial revision
 *
 */

#include	<sys/coherent.h>
#include 	<sys/errno.h>
#include 	<sys/file.h>
#include 	<sys/sb16.h>
#include 	<sys/sound.h>
#include 	<sys/types.h>

/*--------------------------------------------------------------------------*/
/*	SB16MixerIOCTL															*/
/*--------------------------------------------------------------------------*/
typedef struct _SB16MixerIOCTL
{
	int			cmd;				/* ioctl() command 				 */
	uchar_t		registerLeft;		/* register for left channel	 */
	uchar_t		registerRight;		/* register for right channel	 */
	int			levelShift;			/* reduces 8-bit level			 */
	int			leftShift;			/* bit position of left channel	 */
	int			rightShift;			/* bit position of right channel */
	int			bRoot;				/* only superuser may modify	 */
	int			leftShadow;			/* left register shadow			 */
	int			rightShadow;		/* right register shadow		 */
}SB16MixerIOCTL;

/*--------------------------------------------------------------------------*/
/*	Mixer register table													*/
/*--------------------------------------------------------------------------*/
static SB16MixerIOCTL ioctlTable[] =
{
/*                                              bRoot--------------------\	*/
/*												rightShift------------\  |	*/
/*												leftShift----------\  |  |	*/
/*												levelShift------\  |  |  |  */
/* cmd			registerLeft			registerRight			|  |  |  |	*/
{ MXR_MUTE,		0,						0,						0, 0, 0, 0 },
{ MXR_MASTER,	SB16_MXRREG_MASTER_L,	SB16_MXRREG_MASTER_R,	3, 3, 3, 0 },
{ MXR_VOICE,	SB16_MXRREG_VOICE_L,	SB16_MXRREG_VOICE_R,	3, 3, 3, 0 },
{ MXR_MIDI,		SB16_MXRREG_MIDI_L,		SB16_MXRREG_MIDI_R,		3, 3, 3, 0 },
{ MXR_CD,		SB16_MXRREG_CD_L,		SB16_MXRREG_CD_R,		3, 3, 3, 0 },
{ MXR_LINE,		SB16_MXRREG_LINE_L,		SB16_MXRREG_LINE_R,		3, 3, 3, 0 },
{ MXR_MIC,		SB16_MXRREG_MIC,		0,						3, 3, 3, 0 },
{ MXR_PC_SPKR,	SB16_MXRREG_PC_SPKR,	0,						6, 6, 6, 0 },
{ MXR_IN_GAIN,	SB16_MXRREG_IN_GAIN_L,	SB16_MXRREG_IN_GAIN_R,	6, 6, 6, 0 },
{ MXR_OUT_GAIN,	SB16_MXRREG_OUT_GAIN_L,	SB16_MXRREG_OUT_GAIN_R,	6, 6, 6, 1 },
{ MXR_TREBLE,	SB16_MXRREG_TREBLE_L,	SB16_MXRREG_TREBLE_R,	4, 4, 4, 0 },
{ MXR_BASS,		SB16_MXRREG_BASS_L,		SB16_MXRREG_BASS_R,		4, 4, 4, 0 },
{ MXR_AGC,		SB16_MXRREG_AGC,		0,						7, 0, 0, 0 },
{ MXR_OS_LINE,	SB16_MXRREG_OUT_SWITCH,	SB16_MXRREG_OUT_SWITCH,	7, 4, 3, 0 },
{ MXR_OS_CD,	SB16_MXRREG_OUT_SWITCH, SB16_MXRREG_OUT_SWITCH,	7, 2, 1, 0 },
{ MXR_OS_MIC,	SB16_MXRREG_OUT_SWITCH,	0,						7, 0, 0, 0 },
{ MXR_IS_MIDI_L,SB16_MXRREG_IN_L_SWITCH,SB16_MXRREG_IN_L_SWITCH,7, 6, 5, 0 },
{ MXR_IS_MIDI_R,SB16_MXRREG_IN_R_SWITCH,SB16_MXRREG_IN_R_SWITCH,7, 6, 5, 0 },
{ MXR_IS_LINE_L,SB16_MXRREG_IN_L_SWITCH,SB16_MXRREG_IN_L_SWITCH,7, 4, 3, 0 },
{ MXR_IS_LINE_R,SB16_MXRREG_IN_R_SWITCH,SB16_MXRREG_IN_R_SWITCH,7, 4, 3, 0 },
{ MXR_IS_CD_L,	SB16_MXRREG_IN_L_SWITCH,SB16_MXRREG_IN_L_SWITCH,7, 2, 1, 0 },
{ MXR_IS_CD_R,	SB16_MXRREG_IN_R_SWITCH,SB16_MXRREG_IN_R_SWITCH,7, 2, 1, 0 },
{ MXR_IS_MIC,	SB16_MXRREG_IN_L_SWITCH,0,						7, 0, 0, 0 }
};

/*----------------------------------------------------------------------------
|	void
|	sb16mixer_write (uchar_t mixerRegister, uchar_t value)
|
|	mixerRegister	Mixer register to set
|	value			Byte to be written into mixerRegister
|
|	Writes a value to 'mixerRegister'.
----------------------------------------------------------------------------*/
static void
sb16mixer_write (mixerRegister, value)
uchar_t	mixerRegister;
uchar_t	value;
{
	int	s = sphi();	/* sb16intr accesses SB16_MIXER_ADDRESS */

	outb(SB16_MIXER_ADDRESS, mixerRegister);
	outb(SB16_MIXER_DATA, value);

	spl(s);
}

/*----------------------------------------------------------------------------
|	uchar_t
|	sb16mixer_read (uchar_t mixerRegister)
|
|	mixerRegister	Mixer register to read from
|
|	Reads a value from 'mixerRegister'.
|
|	Returns the value read.
----------------------------------------------------------------------------*/
static uchar_t
sb16mixer_read (mixerRegister)
uchar_t	mixerRegister;
{
	uchar_t	value;
	int		s;

	s = sphi();	/* sb16intr accesses SB16_MIXER_ADDRESS */

	outb(SB16_MIXER_ADDRESS, mixerRegister);
	value = inb(SB16_MIXER_DATA);

	spl(s);

	return(value);
}

/*----------------------------------------------------------------------------
|	int
|	sb16mixer_init ()
|
|	Initializes the mixer interface by resetting the mixer chip, and
|	querying the state of all mixer controls.
|
|	Returns 0 on success, or -1 on error.
----------------------------------------------------------------------------*/
int
sb16mixer_init ()
{
	int ix;

	sb16mixer_write(SB16_MXRREG_RESET, 0);

	for (ix = 1; ix < sizeof(ioctlTable) / sizeof(ioctlTable[0]); ++ix)
	{
		int	mask;
		int	reg;
		int	shift;

		mask = (1 << (8 - ioctlTable[ix].levelShift)) - 1;

		/* left */
		reg							= ioctlTable[ix].registerLeft;
		shift						= ioctlTable[ix].leftShift;
		ioctlTable[ix].leftShadow	= (sb16mixer_read(reg) >> shift) & mask;

		/* right */
		reg = ioctlTable[ix].registerRight;

		if (reg != 0)
		{
			shift						= ioctlTable[ix].rightShift;
			ioctlTable[ix].rightShadow	= (sb16mixer_read(reg) >> shift) & mask;
		}
		else
			ioctlTable[ix].rightShadow	= ioctlTable[ix].leftShadow;

		/* adjust to 0-255 range */
		ioctlTable[ix].leftShadow	<<= ioctlTable[ix].levelShift;
		ioctlTable[ix].rightShadow	<<= ioctlTable[ix].levelShift;
	}

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16mixer_ioctl_get (SB16MixerIOCTL *pMixerIoctl, char *data)
|
|	pMixerIoctl		Pointer to an entry in the mixerControl table
|	data			Pointer to a user supplied MixerControl structure
|
|	Returns a mixer control setting from the shadow register.  The mixer
|	chip is not accessed.
|
|	Returns 0 on success, or an error code on failure.
----------------------------------------------------------------------------*/
static int
sb16mixer_ioctl_get(pMixerIoctl, data)
SB16MixerIOCTL	*pMixerIoctl;
char			*data;
{
	MixerControl mixerControl;

	mixerControl.left = pMixerIoctl->leftShadow;

	if (pMixerIoctl->cmd == MXR_AGC)
		mixerControl.left	= (~mixerControl.left) & 0xFF;

	if (pMixerIoctl->registerRight != 0)
		mixerControl.right	= pMixerIoctl->rightShadow;
	else
		mixerControl.right	= mixerControl.left;

	mixerControl.levels	= 1 << (8 - pMixerIoctl->levelShift);
	mixerControl.bRoot	= pMixerIoctl->bRoot;

	kucopy(&mixerControl, data, sizeof(mixerControl));

	return(0);
}

/*----------------------------------------------------------------------------
|	void
|	sb16mixer_set_control (SB16MixerIOCTL *pMixerIoctl,
|						   int leftValue, int rightValue)
|
|	pMixerIoctl		Pointer to an entry in the mixerControl table
|	leftValue		Left channel value (0-255)
|	rightValue		Right channel value (0-255)
|
|	Sets the specified mixer control to the supplied values.
----------------------------------------------------------------------------*/
void
sb16mixer_set_control (pMixerIoctl, leftValue, rightValue)
SB16MixerIOCTL	*pMixerIoctl;
int				leftValue;
int				rightValue;
{
	int	valueMask;
	int	mask;
	int	value;
	int	reg;

	leftValue 	= leftValue >> pMixerIoctl->levelShift;
	rightValue	= rightValue >> pMixerIoctl->levelShift;

	valueMask	= (1 << (8 - pMixerIoctl->levelShift)) - 1;

	/* left */
	reg	= pMixerIoctl->registerLeft;

	if (reg != 0)
	{
		mask	= valueMask << pMixerIoctl->leftShift;
		value	= sb16mixer_read(reg);
		value	= (value & ~mask) | (leftValue << pMixerIoctl->leftShift);
		sb16mixer_write(reg, value);
	}

	/* right */
	reg	= pMixerIoctl->registerRight;

	if (reg != 0)
	{
		mask	= valueMask << pMixerIoctl->rightShift;
		value	= sb16mixer_read(reg);
		value	= (value & ~mask) | (rightValue << pMixerIoctl->rightShift);
		sb16mixer_write(reg, value);
	}
}

/*----------------------------------------------------------------------------
|	int
|	sb16mixer_set_mute (SB16MixerIOCTL *pMixerIoctl,
|						MixerControl *pMixerControl)
|
|	pMixerIoctl		Pointer to an entry in the mixerControl table
|   pMixerControl	Pointer to the kernel's copy of the user's settings
|
|	Turns on/off muting by setting the master volume to 0 or restoring the
|	pre-mute master volume.
----------------------------------------------------------------------------*/
void
sb16mixer_set_mute (pMixerControl)
MixerControl	*pMixerControl;
{
	int	lValue;
	int rValue;

	if (pMixerControl->left == 0)
	{
		lValue	= ioctlTable[1].leftShadow;
		rValue	= ioctlTable[1].rightShadow;
	}
	else
	{
		lValue	= 0;
		rValue	= 0;
	}

	sb16mixer_set_control(&ioctlTable[1], lValue, rValue);
}

/*----------------------------------------------------------------------------
|	int
|	sb16mixer_ioctl_set (SB16MixerIOCTL *pMixerIoctl, char *data)
|
|	pMixerIoctl		Pointer to an entry in the mixerControl table
|   data            Pointer to a user supplied MixerControl structure
|
|	Given that the user has permission to change the setting, this function
|	sets the appropriate mixer register and saves the requested value in
|	a shadow register.
|
|	Returns 0 on success, or EPERM if the user does not have permission
|	to change the selected control.
----------------------------------------------------------------------------*/
static int
sb16mixer_ioctl_set(pMixerIoctl, data)
SB16MixerIOCTL	*pMixerIoctl;
char			*data;
{
	MixerControl mixerControl;

	if (pMixerIoctl->bRoot && !super())
		return(EPERM);

	if (ukcopy(data, &mixerControl, sizeof(mixerControl)) != 0)
	{
		int	bModify	= 1;

		switch (pMixerIoctl->cmd)
		{
			case MXR_AGC:
				mixerControl.left = ~mixerControl.left;
				break;

			case MXR_MUTE:
				sb16mixer_set_mute(&mixerControl);
				break;

			case MXR_MASTER:
				if (ioctlTable[0].leftShadow)
					bModify	= 0;
				break;
		}

		pMixerIoctl->leftShadow	 = mixerControl.left & 0xFF;
		pMixerIoctl->rightShadow = mixerControl.right & 0xFF;

		if (bModify)
			sb16mixer_set_control(pMixerIoctl,
								  pMixerIoctl->leftShadow,
								  pMixerIoctl->rightShadow);
	}

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16mixer_ioctl (int cmd, char *data, mode)
|
|	cmd			IOCTL command
|	data		Pointer to user data
|	mode		Mode the file was open in
|
|	Handles mixer control changes/inquires.
|
|	Returns 0 on success, or an error code on failure.
----------------------------------------------------------------------------*/
int
sb16mixer_ioctl (cmd, data, mode)
int		cmd;
char	*data;
int		mode;
{
	int	ix;

	for (ix = 0; ix < sizeof(ioctlTable) / sizeof(ioctlTable[0]); ++ix)
	{
		if (cmd == (ioctlTable[ix].cmd | MXR_IOCTL_GET))
		{
			/* get operation */
			if (mode & IPR)
				return(sb16mixer_ioctl_get(&ioctlTable[ix], data));
			else
				return(EACCES);
		}

		if (cmd == (ioctlTable[ix].cmd | MXR_IOCTL_SET))
		{
			/* set operation */
			if (mode & IPW)
				return(sb16mixer_ioctl_set(&ioctlTable[ix], data));
			else
				return(EACCES);
		}
	}

	return(ENODEV);
}


