/*
 * LP-11 line printer driver.
 *
 *	A driver for LP-11 type printers utilizing buffers from the buffer pool
 *	rather than the character list.
 *
 *	It is basically tailored to the printers in use at the University
 *	of NSW (CDC-type with home-grown interfaces to emulate LP-11).
 *	This interface converts lower case to upper case (and hence this
 *	driver doesn't), and supports the 12-channel VFU. It does not support
 *	the line-slew special function, as this is so useless anyway.
 *
 *	The data buffer register is 16 bits wide. Normally, 7-bit ascii
 *	is written to the low order 7 bits. Bit 15 enables the special
 *	functions (VFU ENB). For these functions, bit 6 must also be on.
 *	Bit 5 is unimportant. Bit 4 selects channel control (when 0) or
 *	slew mode (when 1). In channel control mode, bits 0 to 3 select
 *	the channel number (0 to 11). In slew mode, bits 0 to 3 contain
 *	the count of linefeeds required (only 0 and 1 are used!)
 */

#include <param.h>
#include <conf.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <file.h>
#include <buf.h>
#include <seg.h>
#include <lp.h>
#include <errno.h>

extern struct user u;
extern char b[];

extern struct lpst lp[];
extern short nlp;

#define	LPPRI	(PZERO+5)	/* sleep priority */
#define	LPSPL	spl4	/* interrupt priority level */

struct lpdev
{
	short	lpsr;
	short	lpbuf;
};

/*
 * status register bits
 */
#define	IENABLE	0100		/* enable interrupts */
#define	READY	0200		/* printer ready */
#define	PARITY	040000		/* parity error/power off - cdc only */
#define	ERROR	0100000		/* error */


/*
 * Enforce single open (for writing), but allow unlimited opens
 * for reading (for mode changes using e.g. slp)
 */
lpopen(dev, flag)
{
	register struct lpst	*plp;
	register int		d;

	if ((d = minor(dev)) >= nlp)	/* illegal */
	{
		u.u_error = ENXIO;
		return;
	}

	if ((flag & FWRITE) == 0)	/* open for reading to get ioctl fd */
		return;

	plp = &lp[d];
	if ((plp->state != CLOSED) || tkword(plp->lpaddr) == 0 ||
	    (plp->lpaddr->lpsr & ERROR))
	{
		u.u_error = ENXIO;
		return;
	}

	plp->state = OPEN;
	plp->icol = plp->ocol = 0;
}

/*
 * Close routine. Return if open for reading, otherwise set flag
 * and let lpint() take care of it.
 */
lpclose(dev, flag)
{
	register struct lpst	*plp;

	if ((flag & FWRITE) == 0)	/* was open for reading */
		return;

	plp = &lp[minor(dev)];
	LPSPL();
	plp->state = CLOSING;
	if (plp->bfwd == NULL)	/* need interrupt to complete close */
		plp->lpaddr->lpsr |= IENABLE;
	spl0();
}

/*
 * Write routine. Grabs a whole buffer per write, hence 512-byte writes
 * are optimally efficient. Uses iomove to move great chunks around.
 */
lpwrite(dev)
{
	register struct buf	*bp;
	register struct lpst	*plp;
	register int		n;

	plp = &lp[minor(dev)];
	while (u.u_count != 0 && u.u_error == 0)
	{
		LPSPL();
		while (plp->bfree == 0)		/* no free buffers */
		{
			plp->flag |= ASLP;
			sleep(plp, LPPRI);
		}
		if (plp->flag & FLSH)		/* flush all output */
		{
			u.u_error = EIO;
			spl0();
			return;
		}
		plp->bfree--;
		spl0();
		bp = geteblk();
		bp->av_forw = NULL;
		n = min(u.u_count, BSIZE);
		bp->b_bcount = n;
		ka5->r[0] = baddr(bp);
		iomove(&b, n, B_WRITE);
		LPSPL();
		if (plp->bfwd == NULL)		/* first buffer in q */
		{
			plp->bfwd = plp->blst = bp;
			plp->ba = &b;
			plp->bc = bp->b_bcount;
			plp->lpaddr->lpsr |= IENABLE;
		}
		else			/* add to end */
		{
			plp->blst->av_forw = bp;
			plp->blst = bp;
		}
		spl0();
	}
}

/*
 * Interrupt handler. Most of the work is done here.
 */
lpint(dev)
{
	register struct lpst	*plp;
	register int		*lpa;
	int *lps;
	int	ka5sav;

	ka5sav = ka5->r[0];
	plp = &lp[minor(dev)];
	/*
	 * When the printer is powered on or off, for a brief moment
	 * the unibus registers disappear, causing bus errors if an
	 * attempt is made to access them!
	 */
	while (tkword(plp->lpaddr) == 0);	/* is anyone home? */
	lps = &plp->lpaddr->lpsr;
	lpa = &plp->lpaddr->lpbuf;
	if (*lps & ERROR)
	{
		if (*lps & PARITY)	/* parity or power off */
		{
			*lps |= ERROR;	/* clear error bit */
			if ((*lps & ERROR) == 0)	/* error cleared - was parity */
			{
				printf("LP%o: parity error\n", minor(dev));
				goto ok;
			}
			if ((plp->flag & POFF) == 0)	/* first time only */
			{
				printf("LP%o: powered off\n", minor(dev));
				plp->flag |= POFF;
			}
		}
		plp->flag |= ERR;
		*lps &= ~IENABLE;
		timeout(lpint, dev, HZ);	/* try again in one second */
		return;
	}

	if ((*lps & READY) == 0)	/* not ready? */
		return;			/* prob. an unneeded timeout */

ok:
	plp->flag &= ~(ERR|POFF);
	if (plp->bfwd == NULL)		/* nowt to do ... */
		goto fin;

	if (plp->flag & FLSH)		/* flush all output */
	{
		register struct buf	*bp;
		struct buf *fred;

		bp = plp->bfwd;
		do
		{
			fred = bp->av_forw;
			brelse(bp);
			plp->bfree++;
		} while (bp = fred);
		plp->bfwd = NULL;
		if (plp->state == OPEN && (plp->flag & ASLP))
		{
			plp->flag &= ~ASLP;
			wakeup(plp);	/* may be sleeping in open or write */
		}
		plp->flag &= ~FLSH;
		goto fin;
	}
	*lps |= IENABLE;	/* not set if came from timeout */
	{
		register struct buf *bp;

		bp = plp->bfwd;
		ka5->r[0] = baddr(bp);
	}
loop:
	while ((*lps & READY) && (plp->bc-- > 0))
	{
		register	c;
		char		cbase;

		cbase = 0;
		c = (*plp->ba++) & 0377;	/* watch sign exten. */
		switch (c)
		{
		case '\t':
			plp->icol = (plp->icol + 8) & ~07;
			break;

		case '\b':
			if (plp->icol > 0)
				plp->icol--;
			break;

		case PERFSKP:			/* skip to perforations */
			c = 0100103;		/* channel 4 */
		case '\f':
		case '\n':
		case '\r':
			plp->icol = 0;
			plp->ocol = 0;
			*lpa = c;
			break;

		case '{':
			cbase = '(';
			goto dflt;
		case '}':
			cbase = ')';
			goto dflt;
		case '`':
			cbase = '\'';
			goto dflt;
		case '|':
			cbase = '!';
			goto dflt;
		case '~':
			cbase = '^';
		dflt:

		default:
			if (c < 040 || c >= 0177)	/* garbage char */
				c = '?';		/* prints thusly */
			if (plp->icol < plp->ocol)	/* overstrike */
			{
				*lpa = '\r';
				plp->ocol = 0;
				plp->ba--;	/* "unget" the character */
				plp->bc++;
				break;
			}
			while (plp->icol > plp->ocol)
			{
				*lpa = ' ';
				plp->ocol++;
			}
			if (cbase)		/* half-ascii overprint */
			{
				*lpa = cbase;		/* base char */
				*(--plp->ba) = '-';	/* plant overprint */
				plp->bc++;
				*lpa = '\r';
				plp->ocol = 0;
				break;
			}
			*lpa = c;
			plp->icol++;
			plp->ocol++;
		}
	}

	if (plp->bc <= 0)	/* this buffer gone */
	{
		register struct buf	*bp;

		bp = plp->bfwd;
		plp->bfwd = bp->av_forw;
		brelse(bp);
		plp->bfree++;
		if (plp->state == OPEN && (plp->flag & ASLP))
		{
			plp->flag &= ~ASLP;
			wakeup(plp);
		}
		if (bp = plp->bfwd)	/* more buffers to go */
		{
			ka5->r[0] = baddr(bp);
			plp->ba = &b;
			plp->bc = bp->b_bcount;
			goto loop;
		}

fin:
		*lps &= ~IENABLE;
		if (plp->state == CLOSING)
		{
			plp->state = CLOSED;
			if (plp->ocol && (*lps & READY))
			{
				*lpa = '\n';
				plp->ocol = 0;
			}
		}
	}
	ka5->r[0] = ka5sav;
}

/*
 * Get/set info to/from user.
 */
lpioctl(dev, cmd, addr, flag)
{
	register struct lpst *plp;
	struct lpiocb lpiocb;

	plp = &lp[minor(dev)];
	if (cmd == LPGET)
	{
		lpiocb.lp_state = plp->state;
		lpiocb.lp_flag = plp->flag;
		if (copyout((caddr_t)&lpiocb, addr, sizeof(lpiocb)))
			u.u_error = EFAULT;
	}
	else if (cmd == LPSET)
	{
		if (copyin(addr, (caddr_t)&lpiocb, sizeof(lpiocb)))
			u.u_error = EFAULT;
		else if (lpiocb.lp_flag & FLSH)	/* may only set this bit */
		{
			LPSPL();
			if (plp->bfwd)
				plp->flag |= FLSH;
			spl0();
		}
	}
	else
		u.u_error = EINVAL;
}
