/*
 * CDC U200 emulator driver using DEC DP11 synchronous interface
 *
 * 4 devices are necessary so that closes are handled correctly:
 *	0	(r)	command down
 *	1	(w)	command up
 *	2	(r)	job down
 *	3	(w)	job up
 */

#include <param.h>
#include <buf.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <file.h>
#include <seg.h>
#include <errno.h>
#include <ei.h>
#include <lp.h>

extern struct user u;
extern char b[];
extern char partab[];	/* for the parity bits */

#define	DPADDR	((struct device *)0174770)
#define	EISPL	spl6
#define	EIPRI	(PZERO+1)	/* sleep priority */
#define	NSYNC	3		/* number of sync characters transmitted */
#define	NMSG	15		/* number of messages that may be stacked */

/*
 * DP11 device registers
 */
struct	device
{
	short	rcsr;		/* receive status register */
#define	STRPSYNC  0000001	/* all input sync characters stripped */
#define	HDUPLX	  0000002	/* half-duplex mode */
/*		  0000004	/* maintenance mode */
#define	RIENB	  0000100	/* interrupt enable */
#define	RDONE	  0000200	/* character received - 8 bits/char default */
#define	ODDPAR	  0010000	/* last character had odd parity */
	char	rbuf;		/* receive buffer */
	char	syn0;		/* sync character register */
	short	xcsr;		/* transmit status register */
/*		  0000001	/* terminal ready */
#define	IDLESYNC  0000002	/* send sync if not handled in time */
#define	SIENABLE  0000040	/* interrupt enable (status) */
#define	XIENB	  0000100	/* interrupt enable */
#define	XDONE	  0000200	/* character sent */
/*		  0002000	/* clear-to-send signal */
#define	CARRIER	  0004000	/* current state of carrier */
#define	DSRDY	  0010000	/* data-set-ready */
#define	RING	  0020000	/* ring signal received */
#define	RORUN	  0040000	/* receive overrun */
#define	CARR_OFF  0100000	/* carrier went off */
	char	xbuf;		/* transmit buffer */
	char	syn1;
};

int	cyflag;			/* main control flags */
#define	RCOPN	000001		/* cyb.cmds open for read */
#define	WCOPN	000002		/* cyb.cmds open for write */
#define	RJOPN	000004		/* cyb.jobs open for read */
#define	WJOPN	000010		/* cyb.jobs open for write */
#define	RRDY	000020		/* message from cyber available */
#define	WMSG	000040		/* waiting for message from cyber */
#define	RUSE	000100		/* message being read - lock buffer */
#define	JRDY	000200		/* another batch of cards ready to go */
#define	WCRD	000400		/* waiting for card buffer to empty */
#define	JFAVL	001000		/* print buffer full */
#define	WPRT	002000		/* waiting for print output */
#define	RDE1	004000		/* 'read e1' sent for card reader  */
#define	CRDY	010000		/* set if cmd queued to cyber */

int	cyeoj;			/* end of job information */
#define	FINJ	01
#define	ENDJ	02

int	cysus;			/* set when output suspended locally */

/*
 * buffer pointers etc
 * The two rcvbuf elements cycle as follows: the current packet is placed
 * in one; if it was a job, it is then switched over with the other rcvbuf
 * element. In this way, one can be in the process of being read by a user
 * program while the other is being written by the current packet.
 * Syncronisation is on the JFAVL flag in cyflag. If the job was a command
 * response, the buffer is simply swapped with the command buffer (cyabp).
 * As buffers are grabbed from the buffer pool, this driver has intimate
 * knowledge of how many buffers will be needed to make up XBUFSIZ etc.
 */
struct rcvbuf
{
	unsigned rlen;
	struct buf *cyrbp0, *cyrbp1;
	char cyrbf[RBUFSIZ - BSIZE - BSIZE];	/* 26 if BSIZE = 512 */
};
struct rcvbuf cyrbuf0, cyrbuf1;
struct rcvbuf *cyrbuf;		/* current receive buffer */
struct rcvbuf *cyrnbuf;		/* next receive buffer */
unsigned cyrbufl;		/* amount received so far */

caddr_t	cyxbufp;		/* next char to be sent */
struct buf *cyjbp0, *cyjbp1;	/* card buffers */
struct buf *cyjbp;		/* current transmit card buffer */
struct buf *cyabp;		/* buffer for command response from cyber */
char cycbuf[CBUFSIZ+PROTOCOL];	/* command to cyber (4 protocol bytes added) */

/*
 * Error information
 */
char	cyerrf;		/* receive error number */
#define	ERR_FMT	0	/* format error in received msg. */
#define	ERR_LPC	1	/* lpc error in received msg. */
#define	ERR_REC	2	/* hardware detected receive error. */
#define	ERR_XMT	3	/* last msg. transmitted no good. */
#define	ERR_LEN	4	/* last msg. received too long. */
#define	ERR_PRO	5	/* talk syncronizing error. */

/*
 * short error message
 */
char	cyerrfm[][3] =		/* 3 chars each - see cyerep */
{
	"fmt", "lpc", "rec", "xmt", "len", "pro",
};

unsigned	cyerrfc[6];	/* error counts */
unsigned	cyering;
unsigned	cyecarrier;
unsigned	cyeoverun;

char	cysite;			/* site address from last msg. */
char	cystation;		/* station address from last msg. */
char	cycode;			/* control code from last msg. */
				/* can be 0,WRT,ALERT,POLL */
char	cyetyp;			/* interrogation code from last msg. */
char	cywstat = THIS_STATION;	/* station address from last write msg. */
char	cytstat;		/* station address to be transmitted */

caddr_t	cymstack[NMSG];		/* a msg stack */
int	cystkp;			/* msg stack pointer */
int	cynsync;
caddr_t	cymsg;			/* ptr to last 'msg' sent to cyber */
caddr_t	cynowmsg;		/* ptr to an immediate msg */

/*
 * Finite state machine stuff
 */
/*
 * receiver state
 */
int	cyrstate;
#define	CR_SOH	1	/* start of head */
#define	CR_SITA	2	/* site number */
#define	CR_STAA	3	/* status */
#define	CR_CC	4	/* control code */
#define	CR_MSG0	5	/* message - carriage control */
#define	CR_MSGE	6	/* message - escape sequence */
			/* poll reader/printer status */
#define	CR_MSG	7	/* message - data */
#define	CR_END	8	/* wait for SOH? */
#define	CR_EOT	9	/* end-of-transmission */
#define	CR_LPC	10	/* handle lpc */
#define	CR_ERR	11	/* only 1 reference!! */

/*
 * transmitter state
 */
int	cyxstate;
#define	CX_SYNC	1	/* send sync codes */
#define	CX_SOH	2	/* send SOH */
#define	CX_SITA	3	/* send site number */
#define	CX_STAA	4	/* send status byte */
#define	CX_MSG	5	/* send message */
#define	CX_LPC	6	/* send lpc */
#define	CX_END	7	/* cleanup */

/*
 * control sequences:
 *
 *	pdp -> cyber
 *		NAK, ERROR, RD, ACK
 *	cyber -> pdp
 *		WRT, CWRT, RWRT, POLL, ALERT
 *
 *	WRT,RWRT,CWRT,ALERT ) -> (ACK
 *	POLL) -> (RD, NAK(station code=0140)
 *
 *	E2 -> poll printer
 *	E3 -> poll reader
 *
 *	CWRT, RWRT  -  followed by 7 SYNC's
 *
 * standard messages for cyber:
 */
char	cyackm[] = {	Q(ACK),	Q(EOT)				};
char	cyrejm[] = {	Q(NAK),	Q(EOT)				};
char	cyerrm[] = {	Q(ERROR),	Q(EOT)			};
char	cyreed[] = {	Q(RD),	RCMD,	Q(ESC),	Q(E1),	Q(EOT)	};
char	cygo[]	 = {	Q(RD),	GCMD,	Q(ESC),	Q(E1),	Q(EOT)	};
char	cycont[] = {	Q(RD),	CCMD,	Q(ESC),	Q(E1),	Q(EOT)	};
char	cye1m[]	 = {	Q(RD),	Q(ESC),	Q(E1),	Q(EOT)		};
char	cye2m[]	 = {	Q(RD),	Q(ESC),	Q(E2),	Q(EOT)		};
char	cye3m[]	 = {	Q(RD),	Q(ESC),	Q(E3),	Q(EOT)		};

/*
 * bcd to ascii conversion
 */
char	cybtoa[] =
{
	'-', 'j', 'k', 'l', 'm', 'n', 'o', 'p',
	'q', 'r', '!', '$', '*', '^', '#', '>',
	'+', 'a', 'b', 'c', 'd', 'e', 'f', 'g',
	'h', 'i', '<', '.', ')', '?', ':', ';',
	':', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', '0', '=','\'','\\', '%', '[',
	' ', '/', 's', 't', 'u', 'v', 'w', 'x',
	'y', 'z', ']', ',', '(', '&', '"', '@',
};

/*
 * ascii to bcd conversion
 */
char	cyatob[] =
{
	0120, 0052, 0136, 0056, 0053, 0116, 0135, 0114,	/*  !"#$%&' */
	0134, 0074, 0054, 0060, 0133, 0040, 0073, 0121,	/* ()*+,-./ */
	0112, 0101, 0102, 0103, 0104, 0105, 0106, 0107,	/* 01234567 */
	0110, 0111, 0100, 0077, 0072, 0113, 0057, 0075,	/* 89:;<=>? */
	0137, 0061, 0062, 0063, 0064, 0065, 0066, 0067,	/* @ABCDEFG */
	0070, 0071, 0041, 0042, 0043, 0044, 0045, 0046,	/* HIJKLMNO */
	0047, 0050, 0051, 0122, 0123, 0124, 0125, 0126,	/* PQRSTUVW */
	0127, 0130, 0131, 0117, 0115, 0132, 0055, 0120,	/* XYZ[\]^_ */
	0120, 0061, 0062, 0063, 0064, 0065, 0066, 0067,	/* `abcdefg */
	0070, 0071, 0041, 0042, 0043, 0044, 0045, 0046,	/* hijklmno */
	0047, 0050, 0051, 0122, 0123, 0124, 0125, 0126,	/* pqrstuvw */
	0127, 0130, 0131, 0120, 0120, 0120, 0120, 0120,	/* xyz{|}~  */
};

/*
 * Cyber command/job queue open
 */
eiopen(dev, flag)
dev_t	dev;
int	flag;
{
	register int	x, y, z;

	y = 0;
	switch(minor(dev))
	{
	case 0:			/* cyber commands down */
		x = RCOPN;
		z = FREAD;
		break;
	case 1:			/* cyber commands up */
		x = WCOPN;
		z = FWRITE;
		break;
	case 2:			/* cyber jobs down */
		x = RJOPN;
		z = FREAD;
		break;
	case 3:			/* cyber jobs up */
		x = WJOPN;
		y = RDE1;
		z = FWRITE;
		break;
	default:
		u.u_error = ENXIO;
		return;
	}
	if ((cyflag & x) || flag != z || cystart())
	{
		u.u_error = ENXIO;
		return;
	}
	cyflag |= x|y;
}

/*
 * Cyber command/job queue close
 */
eiclose(dev)
dev_t	dev;
{
	register int	x;

	switch(minor(dev))
	{
	case 0:			/* cyber commands down */
		x = RCOPN;
		break;
	case 1:			/* cyber commands up */
		x = WCOPN;
		break;
	case 2:			/* cyber jobs down */
		x = RJOPN | JFAVL;
		break;
	case 3:			/* cyber jobs up */
		x = WJOPN | RDE1;
		break;
	}

	cyflag &= ~x;
	if ((cyflag & (WJOPN|RJOPN|WCOPN|RCOPN)) == 0)
	{
		brelse(cyabp);
		brelse(cyjbp0);
		brelse(cyjbp1);
		brelse(cyrbuf0.cyrbp0);
		brelse(cyrbuf0.cyrbp1);
		brelse(cyrbuf1.cyrbp0);
		brelse(cyrbuf1.cyrbp1);
	}
}

/*
 * Cyber command/job queue read
 */
eiread(dev)
dev_t	dev;
{
	register unsigned l;

	EISPL();
	if (minor(dev) == 0)		/* cyber commands down */
	{
		while ((cyflag & RRDY) == 0)
		{
			cyflag |= WMSG;
			sleep((caddr_t)&cyabp, EIPRI);
		}
		cyflag &= ~RRDY;
		cyflag |= RUSE;		/* lock the buffer from changes */
		spl0();
		if (l = u.u_count)
		{
			ka5->r[0] = baddr(cyabp);
			iomove(&b, min(l, cyabp->b_bcount), B_READ);
		}
		cyflag &= ~RUSE;
	}
	else				/* cyber jobs down */
	{
		if (cyeoj & ENDJ)	/* terminate last job */
		{
			cyeoj &= ~ENDJ;
			u.u_error = ENXIO;
			spl0();
			return;
		}
		while ((cyflag & JFAVL) == 0)
		{
			cyflag |= WPRT;
			sleep((caddr_t)&cyrnbuf, EIPRI);
		}
		if (cyeoj & FINJ)
			cyeoj = ENDJ;
		spl0();
		/*
		 * Now pass the job back to the reader.
		 * As this may be spread across up to 2 buffers and
		 * a structure buffer, it is a bit messy ...
		 * It is known that rlen cannot exceed RBUFSIZ bytes.
		 */
		if (l = u.u_count)
		{
			register unsigned n;
			register struct buf *bp;

			l = min(l, cyrnbuf->rlen);
			bp = cyrnbuf->cyrbp0;
			n = min(l, BSIZE);
			ka5->r[0] = baddr(bp);
			iomove(&b, n, B_READ);
			l -= n;
			n = min(l, BSIZE);
			if (n && u.u_error == 0)
			{
				bp = cyrnbuf->cyrbp1;
				ka5->r[0] = baddr(bp);
				iomove(&b, n, B_READ);
				l -= n;
				if (l && u.u_error == 0)
					iomove(cyrnbuf->cyrbf, l, B_READ);
			}
		}
		cyflag &= ~JFAVL;
	}
}

/*
 * Cyber command/job queue write
 */
eiwrite(dev)
dev_t	dev;
{
	EISPL();
	if (minor(dev) == 1)		/* cyber commands up */
	{
		register int c;
		register caddr_t p;

		if ((cyflag & CRDY) || (u.u_count > CBUFSIZ))
			u.u_error = ENXIO;
		else
		{
			p = cycbuf;
			*p++ = Q(RD);
			/*
			 * this doesn't happen often, and is very short,
			 * and is on an odd boundary, and the parity bit
			 * must be removed anyway, so use cpass (yuk!)
			 */
			while ((c = cpass()) > 0)
				*p++ = c & 0177;
			if (u.u_error)
			{
				spl0();
				return;
			}
			*p++ = Q(ESC);
			*p++ = Q(E1);
			*p++ = Q(EOT);
			switch (cycbuf[1])
			{
			case LCMD:		/* login */
			case CCMD:		/* continue */
			case BCMD:		/* bye */
				cysus = 0;
				break;

			case SCMD:		/* suspend */
				cysus = 1;
				break;
			}

			cyflag |= CRDY;
			cyqmsg(cycbuf);
		}
	}
	else				/* cyber jobs up */
	{
		register unsigned l, n;

		while (cyflag & JRDY)
		{
			cyflag |= WCRD;
			sleep((caddr_t)&cyjbp, EIPRI);
		}
		spl0();
		if ((l = u.u_count) == 0)
			return;
		ka5->r[0] = baddr(cyjbp0);
		l = min(l, XBUFSIZ);
		n = min(l, BSIZE);
		iomove(&b, n, B_WRITE);
		if (u.u_error)
			return;
		l -= n;
		if (l)			/* any more ? */
		{
			n = l;
			ka5->r[0] = baddr(cyjbp1);
			iomove(&b, n, B_WRITE);
			if (u.u_error)
				return;
		}
		/*
		 * force the last char to be EOT (it should be anyway)
		 */
		b[n - 1] = Q(EOT);
		EISPL();
		cyflag |= JRDY;
		if (cyflag & RDE1)
		{
			cyqmsg(cyreed);		/* r command */
			cyflag &= ~RDE1;
		}
	}
	spl0();
}

/*
 * place messages for cyber in 'fifo' queue
 */
cyqmsg(mp)
caddr_t	mp;
{
	register int x,	sps;
	
	if (cystkp <= NMSG - 2)
	{
		sps = EISPL();
		for (x = ++cystkp; x > 1; x--)
			cymstack[x - 1] = cymstack[x - 2];
		cymstack[0] = mp;
		splx(sps);
	}
	else
		printf("ei: stack overflow\n");
}

/*
 * called to log errors and optionally tell op
 */
cyerep(en)
int	en;
{
	register caddr_t x;
	register caddr_t y;
	register ka5sav;
	static char linerr[] = LINERR;

	cyerrfc[en]++;
	if (cyflag & RUSE)
		return;
	x = cyerrfm[en];
	y = &linerr[7];
	*y++ = *x++;
	*y++ = *x++;
	*y = *x;
	cyabp->b_bcount = sizeof(linerr) - 1;	/* for the null at the end */
	ka5sav = ka5->r[0];
	ka5->r[0] = baddr(cyabp);
	bcopy((caddr_t)linerr, (caddr_t)&b, sizeof(linerr) - 1);
	ka5->r[0] = ka5sav;
	cyflag |= RRDY;
	if (cyflag & WMSG)
	{
		cyflag &= ~WMSG;
		wakeup((caddr_t)&cyabp);
	}
}

/*
 * process cyber write commands -- called from 
 * receive interrupt routine
 */
cywpro()
{
	if (cymsg == &cyjbp)		/* job just sent is acknowledged */
	{
		cyflag &= ~JRDY;
		if (cyflag & WCRD)
		{
			cyflag &= ~WCRD;
			wakeup((caddr_t)&cyjbp);
		}
	}
	else if (cymsg == cycbuf)	/* else command is acknowledged */
		cyflag &= ~CRDY;
	cymsg = NULL;		/* last 'msg' to cyber was accepted */
	switch (cyetyp)		/* process write according to code */
	{
	case E1:			/* command response */
		if ((cyflag & RUSE) == 0)
		{
			if (cyrbufl > BSIZE)
			{
				printf("ei: msg trunc\n");
				cyrbufl = BSIZE;
			}
			{
				register struct buf *bp;

				bp = cyrbuf->cyrbp0;
				cyrbuf->cyrbp0 = cyabp;
				cyabp = bp;
				bp->b_bcount = cyrbufl;
			}
			cyflag |= RRDY;
			if (cyflag & WMSG)
			{
				cyflag &= ~WMSG;
				wakeup((caddr_t)&cyabp);
			}
		}
		break;

	case E2:	/* print output - if can't accept pretend not ready */
		if ((cyflag & JFAVL) || (cyflag & RJOPN) == 0)
		{
			cynowmsg = cye2m;
			break;
		}
		/*
		 * switch the rcvbufs
		 */
		{
			register struct rcvbuf *dummy;

			dummy = cyrbuf;
			cyrbuf = cyrnbuf;
			cyrnbuf = dummy;
			dummy->rlen = cyrbufl;
		}
		cyflag |= JFAVL;
		if (cyflag & WPRT)
		{
			cyflag &= ~WPRT;
			wakeup((caddr_t)&cyrnbuf);
		}
		if (cystkp == 0)	/* tell cyber want more output */
			cynowmsg = cye3m;
		break;

	case E3:			/* card input (to cyber) */
		if (cyflag & JRDY)
			cynowmsg = &cyjbp;
		else		/* no cards to go so not ready */
		{
			cynowmsg = cye2m;
			cyflag |= RDE1;
		}
		break;
	}
	cyxbufp = cyackm;		/* acknowlege receipt of write */
}

/*
 * Cyber receive interrupt routine
 */
eirint()
{
	register int c;
	register ka5sav;
	static int lpc;

	if ((cyflag & (RJOPN | RCOPN)) == 0)
	{
		DPADDR->xcsr = 0;	/* no-one listening, so why bother? */
		DPADDR->rcsr = 0;
		return;
	}
	ka5sav = ka5->r[0];
	while (DPADDR->rcsr & RDONE)
	{
		c = DPADDR->rbuf & 0177;	/* obtain next char sans parity */
		lpc ^= c;			/* calc longitudinal parity */
		if ((DPADDR->rcsr & ODDPAR) == 0)
			cyrstate = CR_ERR;
		switch (cyrstate)
		{
		case CR_SOH:		/* first char must be start of header */
			if (c != SOH)
				goto cyrerror;
	cyrsoh1:
			lpc = c;		/* only SOH so far */
			cyrstate = CR_SITA;
			break;

		case CR_SITA:		/* second char <=0177 & >=0160  */
			if (c < 0160)
				goto cyrerror;
			cysite = c;
			cyrstate = CR_STAA;
			break;

		case CR_STAA:		/* third char 0140,0141,0160,0161 */
			if ((c & ~0021) != 0140)
				goto cyrerror;
			cystation = c;
			cyrstate = CR_CC;
			break;

		case CR_CC:			/* fourth must be acceptable command */
			switch (c)
			{
			default:
				goto cyrerror;

			case WRT:
			case CWRT:
			case RWRT:
				cycode = WRT;
				cyrstate = CR_MSG0;
				break;

			case POLL:
			case ALERT:
				cycode = c;
				cyrstate = CR_EOT;
				break;
			}
			break;

		case CR_MSG0:		/* process carriage control portion of message */
			cyrstate = CR_MSG;
			switch (c)
			{
			case SPACE3:		/* '-' */
				eiput('\n');
			case SPACE2:		/* '0' */
				eiput('\n');
		/*	case SPACE1:		/* ' ' */

			default:
				if (cyrbufl == 0)
					goto cyrtr;
				c = '\n';
				break;

			case SPACE0:		/* '+' */
				c = '\r'; 
				break;

			case EJECT:		/* '1' */
				c = '\f';
				break;

			case SKIPC3:		/* '6' */
						/* really BOTFP */
			case SKIPC4:		/* '5' */
				c = PERFSKP;	/* really TOPFP */
				break;

			case EJON:		/* 'R' */
				c = AUTEJ;
				break;

			case EJOFF:		/* 'Q' */
				c = NAUTEJ;
				break;

			case ESC:
				cyrstate = CR_MSGE;
				continue;
			}
			goto cyrmsg3;

		case CR_MSGE:		/* process escape sequence */
			switch (c)
			{
			case E1:
			case E2:
			case E3:			/* ending code */
				DPADDR->rcsr &= ~STRPSYNC;
				cyetyp = c;
				cyrstate = CR_EOT;
				break;

			case EOI:
				cyeoj = FINJ; 
			case CEOL:
			case CCR:
				cyrstate = CR_MSG0;
				break;

			default:			/* '0' or ' ' expansion */
				c -= 040;
				if (c >= 3 && c <= 037)	/* ' ' expansion */
					eiput(BLEXP);
				else
				{
					c -= 040;
					if (c>=3 && c<=017)	/* '0' expansion */
						eiput(ZREXP);
					else
						c = ' ';
				}
				cyrstate = CR_MSG;
				goto cyrmsg3;
			}
			break;

		case CR_MSG:		/* process data portion of message */
			if (c == ESC)
			{
				cyrstate = CR_MSGE;
				break;
			}
	cyrtr:
			if (c>HIGHBCD || c<LOWBCD)
				c = ' ';
			else
				c = cybtoa[c - LOWBCD];
	cyrmsg3:
			if (cyrbufl >= RBUFSIZ)
			{
				if (cyerrf < 0)
					cyerrf = ERR_LEN;
				goto cyrerros;
			}
			eiput(c);
			break;

		case CR_EOT:		/* this char must be "eot" */
			if (c != EOT)
				goto cyrerror;
			cyrstate = CR_LPC;
			break;

		case CR_LPC:		/* this char is longitudinal parity */
			if (lpc != 0177 && cyerrf < 0)
				cyerrf = ERR_LPC;
			cyrstate = CR_SOH;		/* ready for next record */

			if (cysite != THIS_SITE)
			{
				DPADDR->syn0 = SYNC;
				DPADDR->rcsr = HDUPLX | STRPSYNC | RIENB;
				DPADDR->xcsr |= SIENABLE;
				break;
			}
			DPADDR->xcsr = 0;	/* terminate input */
			DPADDR->rcsr = 0;	/* terminate input */
			if (cyerrf >= 0)
			{
				cyerep(cyerrf);
				cyerrf = -1;
				cyxbufp = cyerrm;
			}
			else
			{
				c = cycode;
				if (cystation & 01)	/* odd station address */
					c = Q(c);
				switch (c)
				{
				case Q(WRT):	/* save write address */
					cywstat = cystation;
					cywpro();	/* process write */
					break;

				case Q(ALERT):
					cyxbufp = cyackm;
					cyqmsg(cye1m);
					break;

				case POLL:
					if (cymsg == -1)
					{
						cymsg = NULL;
						cyxbufp = cyrejm;
					}
					else if (cymsg != NULL)
					{
						if (cymsg == &cyjbp)
						{
							cyxbufp = &b;
							cyjbp = cyjbp0;
						}
						else
							cyxbufp = cymsg;
						cyerep(ERR_XMT);
					}
					else if (cynowmsg != NULL)
					{
						if ((cymsg = cynowmsg) == &cyjbp)
						{
							cyjbp = cyjbp0;
							cyxbufp = &b;
						}
						else
							cyxbufp = cymsg;
						cynowmsg = NULL;
					}
					else if (cystkp)
						cyxbufp = cymsg = cymstack[--cystkp];
					else
						cyxbufp = cyrejm;
					break;

				default:
					cyxbufp = cyerrm;
					cyerep(ERR_PRO);
					break;
				}
				cytstat = cywstat;
				if (cycode == POLL && cyxbufp == cyrejm)
					cytstat &= ~01;
			}
			cyrbufl = 0;
			DPADDR->xcsr = XIENB | IDLESYNC;
			DPADDR->rcsr = HDUPLX;
			DPADDR->xbuf = SYNC;
			break;

		case CR_ERR:
	cyrerror:
			if (cyerrf < 0)
				cyerrf = ERR_FMT;
	cyrerros:
			DPADDR->rcsr &= ~STRPSYNC;
			cyrstate = CR_END;
			break;

		case CR_END:		/* look for end of message */
			if (c == SOH)
			{
				cyrbufl = 0;
				goto cyrsoh1;
			}
			break;
		}
	}
	ka5->r[0] = ka5sav;
}

/*
 * Place received character in appropriate buffer
 */
eiput(ch)
char ch;
{
	register unsigned n;

	n = cyrbufl++;
	if (n < BSIZE)
	{
		ka5->r[0] = baddr(cyrbuf->cyrbp0);
		b[n] = ch;
		return;
	}
	n -= BSIZE;
	if (n < BSIZE)
	{
		ka5->r[0] = baddr(cyrbuf->cyrbp1);
		b[n] = ch;
		return;
	}
	n -= BSIZE;
	if (n < RBUFSIZ - 2 * BSIZE)
		cyrbuf->cyrbf[n] = ch;
	else
		cyrbufl--;	/* overrun picked up in eirint */
}

/*
 * Cyber transmit status interrupt routine
 *	cyxbufp points to the message to be sent to the cyber
 *	Q(EOT) signals end of message
 *	any char with parity bit on is not translated
 */
eixint()
{
	register int c;
	static int lpc;

	if ((c = DPADDR->xcsr) & SIENABLE)	/* receive error */
	{
		if (c & (RORUN | RING | CARR_OFF))
		{
			if (cyerrf < 0)
				cyerrf = ERR_REC;
			if (c & RORUN)
				cyeoverun++;
			if (c & RING)
				cyering++;
			if (c & CARR_OFF)
				cyecarrier++;
		}
		DPADDR->xcsr &= ~(CARR_OFF | RORUN | RING);
		DPADDR->syn0 = SYNC;		/* reset just in case */
		return;
	}
	switch (cyxstate)		/* transmit message sequence */
	{
	case CX_SYNC:		/* send 4 syncs */
		c = SYNC;
		if (--cynsync == 0)
			cyxstate = CX_SOH;
		break;

	case CX_SOH:		/* 1st char is start of header */
		c = SOH;
		cyxstate = CX_SITA;
		lpc = 0;
		break;

	case CX_SITA:		/* 2nd char is site address */
		c = cysite;
		cyxstate = CX_STAA;
		break;

	case CX_STAA:		/* 3rd char is station address */
		c = cytstat;
		cyxstate = CX_MSG;
		break;

	case CX_MSG:		/* output data till eot found */
		if (cymsg == &cyjbp)
		{
			register ka5sav;

			ka5sav = ka5->r[0];
			if (cyxbufp == &b[BSIZE])
			{
				cyjbp = cyjbp1;
				cyxbufp = &b;
			}
			ka5->r[0] = baddr(cyjbp);
			c = *cyxbufp++;
			ka5->r[0] = ka5sav;
		}
		else
			c = *cyxbufp++;
		if (c & 0200)
		{
			c &= 0177;
			if (c == EOT)		/* end of record */
				cyxstate = CX_LPC;
		}
		else
			c = (c < ' ') ? 0120 : cyatob[c - ' '];
		break;

	case CX_LPC:		/* output long. parity */
		c = ~lpc & 0177;
		cyxstate = CX_END;
		break;

	case CX_END:		/* all msg sent - tidy up */
		if (cyflag & (WJOPN|RJOPN|WCOPN|RCOPN))
		{
			DPADDR->syn0 = SYNC;		/* just in case */
			DPADDR->rcsr = HDUPLX | STRPSYNC | RIENB;	/* enable receive */
			DPADDR->xcsr = SIENABLE;	/* enable receive */
		}
		else	/* no open devices so close down */
		{
			DPADDR->rcsr = 0;
			DPADDR->xcsr = 0;
		}
		cyrstate = CR_SOH;
		cyxstate = CX_SYNC;
		cynsync = NSYNC;
		return;
	}
	c |= ~partab[c] & 0200;		/* odd parity gen */
	lpc ^= c;			/* longitudinal parity */
	DPADDR->xbuf = c;
}

/*
 * Initiate receive on DP11 if necessary
 */
cystart()
{
	extern cyawful();

	if ((DPADDR->xcsr & DSRDY) == 0)
	{
		printf("ei: modem not ready\n");
		return(1);
	}
	if ((cyflag & (WJOPN|RJOPN|WCOPN|RCOPN)) == 0)
	{
		DPADDR->syn0 = SYNC;
		DPADDR->rcsr = HDUPLX | STRPSYNC | RIENB;
		DPADDR->xcsr = SIENABLE;
		/*
		 * reinit necessary variables
		 */
		cysus = 0;
		cyflag = 0;
		cystkp = 0;
		cyerrf = -1;
		cymsg = -1;
		cyrbuf = &cyrbuf0;
		cyrnbuf = &cyrbuf1;
		cyrbufl = 0;
		cyrstate = CR_SOH;
		cyxstate = CX_SYNC;
		cynsync = NSYNC;
		cynowmsg = NULL;
		/*
		 * grab some buffers from the buffer pool
		 * (assume there are buffers to burn - else deadlock)
		 */
		cyabp = geteblk();
		cyjbp0 = geteblk();
		cyjbp1 = geteblk();
		cyrbuf0.cyrbp0 = geteblk();
		cyrbuf0.cyrbp1 = geteblk();
		cyrbuf1.cyrbp0 = geteblk();
		cyrbuf1.cyrbp1 = geteblk();
		/*
		 * arrange to poke the cyber every so often
		 */
		timeout(cyawful, (caddr_t)NULL, 20*HZ);
	}
	return(0);
}

/*
 * The cyber goes away if you reject a print output -
 * export will assume that the printer is down and suspend you.
 * Hence the stream of C messages.
 */
cyawful()
{
	if (cyflag & (WJOPN|RJOPN|WCOPN|RCOPN))
	{
		if (cystkp < 2)
		{
			if (cysus == 0)
				cyqmsg(cycont);		/* c command */
			cyqmsg(cygo);			/* g command */
		}
		timeout(cyawful, (caddr_t)NULL, 20*HZ);
	}
}
