;;;	/ p o p / u s r / l i b / e l i z a . p
;;;	Sussex Mini ELIZA programme
;;;	Written by Steven Hardy 18th October 1976
;;;	modified by Aaron Sloman 17 May 1978
;;; the function changeperson is now called before any tests are carried out,
;;; so that you always refers to the user, I to the computer, etc.,
;;; in the transformed sentence, which is then analysed by other procedures
;;; trying to react to it.
;;; The variable "sentence" is local to the function "Eliza", and used non-
;;; locally by other procedures. Thus, general purpose matching procedures
;;; can be defined which simply take a pattern as argument. Examples are
;;; the procedures:  itmatches, itcontains, ithasoneof, itslikeoneof,
;;;	and itsaquestion,
;;; which are used by lots of other procedures to test the current sentence.

;;; Initialise ranseed so that calls of random behave differently on different
;;; occasions.
popmess([%Getpid%]) ->ranseed;

vars sentence;

;;; a table, and some procedures for transforming the input sentence
;;; so that "I" becomes "you", etc. A minor problem is coping with
;;; "are". "you are" should become "i am", whereas in "we are", "they are"
;;; "are" should be left unaltered.
;;; a further difficulty is deciding whether "you" should become "I" or "me".
;;; This program uses the simple test that I at the end of the sentence is
;;; unacceptable.
;;; The transformation goes in three stages.
;;;	first find second person occurrences of "are" and mark them.
;;;	then transform according to the table below,
;;;	then replace final "I" with "me".
vars persontransform;
	[[i you]
	[you i]
	[my your]
	[yourself myself]
	[myself yourself]
	[your my]
	[me you]
	[mine yours]
	[yours mine]
	[am are]
	[Are am]
	[we you]]		;;; not always safe!
	-> persontransform;


function lookup(word, table);
	;;; Return the original word if there isn't an entry in the table.
	if	table == []
	then	word
	elseif	word == hd(hd(table))
	then	hd(tl(hd(table)))
	else	lookup(word, tl(table))
	close
end;


vars itcontains itmatches itslikeoneof;		;;; defined below: used in changeperson.

function changeperson(sentence) => sentence;
	vars list1 list2;
	;;; first distinguish second person versions of "are"
	if	not(itcontains("you"))
	then	sentence
	elseif	itmatches([??list1 you are ??list2])
	then	[^^list1 you Are ^^list2]
	elseif	itmatches([??list1 are you ??list2])
	then	[^^list1 Are you ^^list2]
	else sentence
	close -> sentence;
	;;; now transform according to persontransform, defined above.
	maplist(sentence,
		   lambda wd;
			lookup(wd,persontransform)
		   end)
			-> sentence;
	;;; Now change "I" at the end to "me".
	if itmatches([??list1 i]) then [^^list1 me] ->sentence close
end;



;;;	****	READING IN A SENTENCE    ****

;;; The function readsentence below is derived from the library program readline.
;;; it ignores string quotes, e.g. as typed in "don't", "isn't", etc.
;;; it also asks you to type something if you type a blank line.
;;; It uses function changeperson to transform the sentence.
;;; It also strips off "well" and other redundant leading words.
;;; finally it checks if you wish to restore normal error handling (which is
;;;	switched off in the function Eliza) or wish to stop.


vars cucharin; charin -> cucharin;		;;; used in readsentence


vars Uptolow;
`a - `A -> Uptolow;	;;; Used in function lowercase.
function lowercase char => char;
	;;; Used to transform upper to lower case in function readsentence.
	if `A =< char and char =< `Z then char+Uptolow ->char close
end;


function readsentence()=>sentence;
	vars proglist item char sentenceread;
	incharitem(
		lambda ();
			cucharin() -> char;
			loopif	char == 0
			or	char == `'
			then	cucharin() -> char
			close;
			if	char == `\n
			then	termin
			elseif	char==termin
			then	setpop()
			elseif	char == `;
			or	char == `.
			then	` 		;;; return space character.
			else	lowercase(char)
			close
		end)
	-> proglist;
	false -> sentenceread;
	until	sentenceread
	then
		[%until (readitem() -> item, item == termin) then
				item
		  close%] ->sentence;
		if	sentence==[]
		then	ppr('please type something\n');
		else	true -> sentenceread
		close
	close;
	;;; get rid of "well"  and similar redundant starting words
	loopif	itslikeoneof([[well = ==][but = ==]
				[however = ==][and = ==]
				[yes = ==][no = ==]])
	then	tl(sentence) ->sentence;
		if	hd(sentence)==", "
		then	tl(sentence) ->sentence;
		close
	close;
		if	sentence=[debug]
		then	ppr('changing errfun\n');
			syserr ->errfun;
			readsentence() -> sentence;
		elseif	itslikeoneof([[bye][good bye][goodbye]])
		then	setpop()
		close;
	changeperson(sentence) ->sentence
end;



;;;	**** CIRCULATING LISTS OF POSSIBILITIES ****

;;; The next function is used to get the first element of a list, then
;;; put it on the end of the list, so that next time a different element
;;; will be the first one. This enables a stock of standard replies to
;;; be used in certain contexts without it being too obvious.
;;; an alternative would be to use function oneof, but it cannot be
;;; relied on not to be repetitive!
function firstolast(list) => first list;
	;;; use the original list links, to minimise garbage collection.
	vars list1 last;
	hd(list) ->first;
	list -> list1;
	tl(list) ->> list ->last;
	[] -> tl(list1);
	until	tl(last) == []
	then	tl(last) ->last;
	close;
	list1 ->tl(last)
end;




;;;	*****	A COLLECTION OF MATCHING AND RECOGNISING FUNCTIONS   ****

function itmatches(List);
	;;; use capital L to prevent clash of variables inside match.
	match(List,sentence)
end;

function itcontains(x);
	if	atom(x)
	then	member(x,sentence)
	else
		match([== ^^x ==], sentence)
	close
end;

;;; the function ithasoneof takes a list of words and checks whether
;;; the current sentence contains one of them
function ithasoneof(list);
	if	list ==[]
	then	false
	else	itcontains(hd(list))
		or	ithasoneof(tl(list))
	close
end;

function itslikeoneof(List);
	until	List ==[]
	then	if	itmatches(hd(List))
		then	true;	return
		else tl(List) -> List
		close
	close;
	false
end;



;;;	****	RULES FOR REACTING TO THE SENTENCE ****

;;; First we define a macro called newrule.
;;; It works exactly like "function", i.e. it defines a function.
;;; The only difference is that it makes sure the name of the function is
;;; added to the global list functionlist.
;;; This list of function names is repeatedly shuffled and then the
;;; corresponding functions tried in order to see if one of them can
;;; produce a response to the sentence.
;;; if it produces a response other than false, then the response will be
;;; used in replyto. If there is no response then the result of the function try
;;; defined below, will be false, so replyto will try something else.

vars functionlist;
	[] -> functionlist;

macro newrule;
	vars name x;
	itemread() ->name;
	if	name="function"
	then	itemread() ->name
	elseif	identprops(name) = "syntax"
	then	error(name,1,'missing name in newrule')
	close;
	itemread() ->x;
	if	x="("
	then	erase(itemread())
	elseif	x /= ";"
	then	error(x,1,'bad syntax in newrule')
	close;
	unless	member(name, functionlist)
	then	name :: functionlist -> functionlist;
	close;
	dl([function ^name();])
end;


vars problem newproblem;	;;; used to remember something said earlier,
				;;; to be repeated when short of something to say
				;;; Altered in some of the rules, below.

newrule i_think;
	if	itmatches([== i think ==])
	then	'we were discussing you not me'
	close
end;


newrule youthink;
	vars list1;
	if	itmatches([you think ??list1])
	then	['why do you think' ^^list1 ?]
	close;
	sentence -> newproblem;
end;

vars questionlist;
	['perhaps you already know the answer to that question?'
	'is that question important to you?'
	'why exactly do you ask?'
	'do you really want to know?'
	'what makes you think I know the answer?'
	'perhaps you ask questions to cover something up?'] ->questionlist;

newrule itsaquestion;
	if	member(hd(sentence),
			[ did didnt do dont does doesnt were werent
			  will wont would wouldnt could couldnt
			  is isnt are arent am
			  shall shant can cant
			  which why where who
			  what when how])
	or	hd(rev(sentence)) == "?"
	then	firstolast(questionlist) ->questionlist;
			;;; leaves first element of questionlist on the stack.
	else false
	close
end;

newrule family;
	if	ithasoneof([mother father brother sister daughter wife husband son])
	then	oneof(['tell me more about your family''do you like your relatives?'])
	close
end;

vars shortlist;
	['you are being somewhat short with me'
	'perhaps you dont feel very talkative today?'
	'could you be more informative?'
	'are you prepared to elaborate?'
	'I dont think you really trust me'
	'in order to help, I need more information'
	'well?'] ->shortlist;

newrule short;
	if	length(sentence) < 3
	and	not(itsaquestion())
	then	firstolast(shortlist) ->shortlist;
	close
end;

newrule because;
	if	itcontains("because")
	then	'is that the real reason?'
	close
end;


newrule suppnot;
	vars list1 list2 sentence;
		if	hd(sentence)=="because"
		then	tl(sentence) ->sentence
		close;
			;;; That prevents some awkwardness in replies.

	if	itsaquestion()
	or random(100) > 50
	then	false
	elseif	itslikeoneof([[??list1 is not ??list2])
			[??list1 isnt ??list2]
		 	[??list1 are not ??list2]
			[??list1 arent ??list2]
			[??list1 am not ??list2]])
	then
		[suppose ^^list1 were ^^list2]
	elseif	random(100) > 30
	and	itmatches([you are ??list1])
	then	oneof([['how does it feel to be' ^^list1 ?]
			['are you sure you really are' ^^list1 ?]
			['is this the first time you have been' ^^list1 ?]
			['does anyone else know you are' ^^list1?]
			'is that connected with your reason for talking to me?'
			'would you prefer not to be?'
			'do you know anyone else who is?'])
	elseif	itslikeoneof([[??list1 is ??list2]
			[??list1 are ??list2]
			[??list1 am ??list2]])
	then
		[suppose ^^list1 'were not' ^^list2]
	elseif	itslikeoneof([[??list1 do not ??list2]
			[??list1 does not ??list2]
			[??list1 doesnt ??list2]
			[??list1 dont ??list2]])
	then
		[suppose ^^list1 did ^^list2]
	elseif	itslikeoneof([[??list1 do ??list2]
			[??list1 does ??list2]])
	then
		[suppose ^^list1 did not ^^list2]
	elseif	itmatches([??list1 did not ??list2])
	then	[suppose ^^list1 had ^^list2?]
	elseif	itmatches([??list1 did ??list2])
	then	[suppose ^^list1 had not ^^list2 ?]
	close
end;

newrule computer;
	if	ithasoneof([computer computers machine machines robots])
	then	oneof(['do machines worry you?'
		'how would you react if machines took over?'
		'what do you really think of computers?'])
	close
end;

newrule emphatic;
	if	itmatches([of course == ])
	then	'would everyone find that obvious?'
	elseif	ithasoneof([indeed very extremely])
	and	not(itsaquestion())
	then	'are you sure you are not being dogmatic?'
	close
end;

newrule sayitback;
	if	random(100) < 10
	and	not(itsaquestion())
	then	sentence
	close
end;

newrule youarenot;
	if	itmatches([you are not ??list])
	then	['would you be happier if you were' ^^list]
	close
end;

newrule earlier;
	if	random(100) < 20
	then	['earlier you said'] <> problem;
		newproblem -> problem;
		sentence -> newproblem
	close;
end;

newrule every;
	vars list sentence;
	if	itmatches([because ??list])
	then	list -> sentence
	close;

	if	itslikeoneof([[everybody ??list][everyone ??list]])
	then
		['who in particular' ^^list ?]
	elseif	ithasoneof([everyone everybody])
	then	'anyone in particular?'
	elseif	itmatches([nobody ??list])
	then	['are you sure there isnt anyone who' ^^list?]
	elseif	itcontains("every")
	then	'can you be more specific?'
	elseif	itslikeoneof([[== someone ==] [== somebody ==]
			[== some one ==] [== some people ==]
			[== some men ==] [== some women ==]
			])
	then	'who in particular?'
	elseif	itcontains("some")
	then	'what in particular?'
	elseif	itcontains("everything")
	then	'anything in particular?'
	close
end;

newrule mood;
	if	ithasoneof([depressed miserable sad unhappy lonely confused ill unwell])
	then	'do you think the health centre might be able to help?'
	elseif	ithasoneof([happy happier enjoy enjoyment joy pleasure pleased delighted])
	then	'do you think pleasures should be shared?'
	elseif	ithasoneof([hate love hates loves])
	then	'do strong feelings disturb you?'
	close
end;

newrule hippy;
	if	ithasoneof([drugs pot weird groovy freak])
	then	oneof(['far out, man''mind blowing''are you some kind of freak?'])
	close
end;

newrule fantasy;
	vars list;
	if	itslikeoneof([[you are ??list me]
				[i am ??list you]])
	then	oneof([['perhaps in your fantasy we are' ^^list each other?]
			['do you think we should be' ^^list each other?]
			['do you know many people who' ^^list 'each other?']])
	elseif	itslikeoneof([[you ??list me][i ??list you]])
	then	oneof([[perhaps in your fantasy we ^^list each other?]
			[do you think its wrong for people to ^^list each other?]
			[do you think our relationship is too complicated?]
			[is it good that people should ^^list each other?]])
	close
end;


newrule health;
	if	itcontains([health centre])
	or	itcontains([health center])
	or	ithasoneof([doctor psychiatrist therapist therapy])
	then
		oneof(['do you think doctors are helpful?'
			'do you trust doctors?'])
	elseif	ithasoneof([smoke smokes smoking smoker smokers cigarette cigarette cigars])
	then	'smoking can damage your health'
	close
end;

newrule should;
	vars list1 list2;
	if	itsaquestion()
	then	false
	elseif	itslikeoneof([[??list1 should not??list2]
				[??list1 shouldnt ??list2]])
	then	[why shouldnt ^^list1 ^^list2 ?]
	elseif	itmatches([??list1 should ??list2])
	then	[why should ^^list1 ^^list2?]
	close
end;

newrule looks;
	if	ithasoneof([seems seem appears looks apparently])
	then	'appearances can be deceptive'
	close
end;

newrule unsure;
	if	ithasoneof([perhaps maybe probably possibly])
	then	'you dont sound very certain about that'
	close
end;



;;;	****	PROCEDURES FOR SHUFFLING THE LIST OF RECOGNISERS    ****


function delete(item, list);
	if	item == hd(list)
	then	tl(list)
	else	[%hd(list)%] <> delete(item, tl(list))
	close
end;

function shuffle(list);
	if	list == []
	then	[]
	else	vars temp;
		oneof(list) -> temp;
		temp :: shuffle(delete(temp, list))
	close
end;


;;; 	****	THE CONTROLLING PROCEDURES   ****

;;; The macro eliza simply enables you to execute the function Eliza without
;;; typing parentheses, etc.

;;; The function Eliza, once called, retains control until you type CTRL-D,
;;; or say "bye", "goodbye", etc.
;;; It redefines the function errfun to ensure that the user never gets pop11
;;; error messages, but simply has a chance to try again.
;;; Since this can make debugging difficult, it can be undone inside readsentence, by typing debug.

;;; Eliza repeatedly calls the function readsentence and then asks the function
;;; replyto to try the rules on functionlist to see if one of them produces a
;;; response (i.e. something other than false).

;;; However, the very first utterance by the user is treated differently.

vars level;
3 -> level;

vars desperatelist;
		['sorry I dont understand'
		'do go on'
		'what does that suggest to you?'
		'it sounds as if you have real problems'
		'have you talked to anyone else about your problems?'
		'do you really think I can help you?'
		'could you make yourself clearer please?'
		'this sort of discussion can get us into deep water']-> desperatelist;

shuffle(desperatelist) -> desperatelist;	;;; different on different runs.

function desperateanswer();
	;;; used to produce a reply when all else fails
	firstolast(desperatelist) ->desperatelist;
	sentence -> newproblem;
end;

function try(word);
	;;; this is used in replyto to see if executing the value of the word
	;;; leaves anything on the stack. If so it will be used as the answer.
	;;; if not, the answer is false
	vars sl;
	.stacklength ->sl;
	apply(valof(word));
	if .stacklength = sl
	then false
	close;
end;

function replyto(funclist);
	vars answer level;
	if	funclist == []
			;;; got to end of functionlist. try again if level > 1
	then	if	(level - 1 ->> level) > 1
		then	replyto(functionlist)
		else	desperateanswer()
		close
	elseif	(try(hd(funclist)) ->> answer)
	then	answer
	else	replyto(tl(funclist))
	close
end;


function Eliza;
	vars problem functionlist sentence answer;
	function errfun x;
		repeat stacklength() times
			erase()
		close;
		ppr('somethings gone wrong please try again');
		pr(newline);
		rev(functionlist) ->funclist;
		readsentence();
		exitfrom(readsentence);
	end;
	ppr('good day what is your problem');
	pr(newline);
	readsentence() -> problem;
	problem -> sentence;
	if	(itsaquestion()->> answer)
	then	answer
	elseif	length(problem) > 3
	and	random(100) < 15
	then	problem
	else	desperateanswer()
	close .ppr;
	pr(newline);
	loopif true
	then	readsentence() -> sentence;
		ppr(replyto(functionlist));
		pr(newline);
		shuffle(functionlist) -> functionlist;
	close
end;
macro eliza;
	dl([Eliza();])
end;

ppr('\nPlease type\n\teliza\n');
