/* FLOATING POINT OVERFLOW FAULT HANDLER.
 * UNIVERSITY OF NEW MEXICO.
 * This set of routines handles floating faults for most instructions
 * that can generate them. Two notable exceptions are EMOD and POLY.
 * These two instructions are not generated by the C compiler however.
 * The routines are written generically enough that it would be easy
 * to modify the code to handle all possible arithmetic exceptions on
 * VAX 11/730s, 11/750s and 11/780's (after ECO #7).  Before DEC's
 * Engineering Change Order #7 * VAX 11/780s trapped on these errors so
 * they must be handled in a TOTALLY different manner.
 *
 *  To use these routines from Fortran:
 *
 *      call ofault(nummsg)  will initialize the fault handler and set
 *                           the count for number of messages to be
 *                           printed to  nummsg .
 *
 *      k = kfault(dummy)    returns a count of the number of overflows
 *                           and divisions by zero seen so far.
 *
 *      The result of all floating point expononent overflows and
 *      floating point divisions by zero will be the largest possible
 *      floating point number, which is approximately 1.7e+38. Floating
 *	underflows (if you have the fault enabled) will return the smallest
 *	floating point number.
 *
 *  The strategy goes like this: When VAX detects a fault some
 * registers, the PC of the offending instruction and the PSL are saved on the
 * user stack. When we get this we go through all the operand specifiers
 * performing address calculation in a manner compatible with the machine.
 * What this means is that auto-increment and its kin are performed on the
 * register contents pushed on the stack. When we find the destination
 * address of the offending instruction, the maximum value for the data
 * type is placed there for an error return. Our working PC now points to
 * the next instruction past the bad one. We load the saved PC on the stack
 * with our working PC and return. Thus the instruction really never gets
 * executed BUT we maintain consistency with pointers, etc in the code.
 *
 * One might notice that little discrepancies in the addressing modes, like
 * the fact that you aren't supposed to use general mode with the PC, are
 * not checked for. This is so that the code runs a bit quicker. For most
 * of these VAX will give a different fault long before attempting the
 * arithmetic and the others the user is warned against.
 *
 *	Lee Ward, University of New Mexico,	7/2/83.
 *	Inspired by Gary Klimowicz's pre ECO #7 trap handling routines.
 */

#include <signal.h>
#include <stdio.h>
#include "ofault.h"

#ifndef FPE_FLTOVF_FAULT
#define FPE_FLTOVF_FAULT K_FLTOVF
#define FPE_FLTDIV_FAULT K_FLTDIV
#define FPE_FLTUND_FAULT K_FLTUND
#endif

#define	size_of(x)	(x)

/* GLOBALS */
char	*pc;				/* PC to be used by the routines */
long	*regs0t6;			/* Saved at interrupt time */
long	*regs7t11;			/* We cause these to be pushed */
int	num_msgs;			/* Maximum number of error messages */
int	kfault;				/* Overflows detected */

/* ofault_	Set up to trap arithmetic exceptions reported by SIGFPE.
 *		Calling this function a second (or third or ...) time
 *		will cause the number of error messages to be printed
 *		to be reset.
 */
ofault_(nummsgs)
int	*nummsgs;
{
	extern fp_except();
	extern int num_msgs;
	extern int kfault;

	num_msgs = *nummsgs;
	kfault = 0;
	signal (SIGFPE, fp_except);
}

/* kfault_	Return the number of overflows encountered
 */
int kfault_ ()
{
	extern int kfault;

	return (kfault);
}

/* fp_except	Handle floating point faults
 *
 * A little expalnation here as this is somewhat confusing. At interrupt
 * time the hardware pushes registers 0 through 6 on the users stack. These
 * can be found at &signo - 8. The others we arrange to have pushed into
 * the local call frame with the below register definitions. It is important
 * that these are not changed. Note that fault_pc and psl are effectively
 * passed by reference as the REI instruction causes them to be reloaded.
 * It is possible to play with these a little. You just must be careful
 * not to put the processor into an "undefined state." For reference see
 * the Architecture handbook (REI instruction).
 */
fp_except (signo, f_type, myaddr, fault_pc, psl)
int		f_type;
unsigned	psl;
char		*fault_pc;
{
	long first_local[1];		/* MUST be first! */
	register int a, b, c, d, e;	/* Don't change! */
	extern int num_msgs;
	extern int kfault;

	/* Get info off of stack */
	regs7t11 = &first_local[0];
	regs0t6 = &signo - 8;
	pc = fault_pc;

	/* Two switches here. One to print an appropriate message and one
	 * to handle the fault.
	 */
	if (kfault < num_msgs)
		switch (f_type) {
		    case FPE_FLTOVF_FAULT:
			fprintf (stderr, "Floating overflow!\n");
			break;
		    case FPE_FLTDIV_FAULT:
			fprintf (stderr, "Floating division by zero!\n");
			break;
		    case FPE_FLTUND_FAULT:
			fprintf (stderr, "Floating underflow!\n");
			break;
		    default: /* Let the handler switch deal with this */
			break;
		}

	switch (f_type) {
	    case FPE_FLTOVF_FAULT:
	    case FPE_FLTDIV_FAULT:
		flt_ov_fault();
		break;
	    case FPE_FLTUND_FAULT:		/* You have to enable first */
		flt_und_fault();
		break;
	    default:
		err ("Cannot handle fault 0X%x", f_type);
	}
	++kfault;

	fault_pc = pc;
	signal (SIGFPE, fp_except);

	if (kfault == num_msgs)
		err ("No more overflow messages will be printed");

	return;		/* Ala REI */
}

/* flt_ov_fault	Handle a floating overflow fault. A misnomer as we handle
 * division by zero in the same manner.
 */
flt_ov_fault ()
{
	int opcode;
	int x;
	int dest;
	int type;
	char *addr;
	extern char *pc;
	extern ANYTYPE *fetch();
	extern char *addr_reg();
	extern char *opnd_addr();

	opcode = (int )fetch()->byte;
	pc += 1;

	dest = dest_opnd(opcode);
	for (x = 1; x <= num_opnd(opcode); ++x) {
		if ((type = get_type(opcode, x)) == GARBAGE)
			err ("data type of operand? (op = 0X%x, operand %d)"
				, opcode, x);
		addr = opnd_addr (type);
		if (x == dest)
			load (addr, type, MAXFLOAT, MAXDOUBLR);
	}
}

/* flt_und_fault	Handle a floating underflow fault.
 */
flt_und_fault ()
{
	int opcode;
	int x;
	int dest;
	int type;
	char *addr;
	extern char *pc;
	extern ANYTYPE *fetch();
	extern char *addr_reg();
	extern char *opnd_addr();

	opcode = (int )fetch()->byte;
	pc += 1;

	dest = dest_opnd(opcode);
	for (x = 1; x <= num_opnd(opcode); ++x) {
		if ((type = get_type(opcode, x)) == GARBAGE)
			err ("data type of operand? (op = 0X%x, operand %d)"
				, opcode, x);
		addr = opnd_addr (type);
		if (x == dest)
			load (addr, type, MINFLOAT, MINDOUBLR);
	}
}

/* dest_opnd	Given an opcode return the number of the operand specifier
 *		 that is the destination.
 */
dest_opnd (opcode)
int	opcode;
{

	switch (opcode) {

	    case CVTDF:
	    case ADDD2:
	    case SUBD2:
	    case MULD2:
	    case DIVD2:
	    case ADDF2:
	    case SUBF2:
	    case MULF2:
	    case DIVF2:	return (2);
	    case ADDF3:
	    case SUBF3:
	    case MULF3:
	    case DIVF3:
	    case ADDD3:
	    case SUBD3:
	    case MULD3:
	    case DIVD3:	return (3);
	    default:
		err ("cannot determine destination operand (opcode = %x)");
		return (-1);
	}
}

/* num_opnd	Given an opcode return the number of operand specifiers
 *		 associated with it.
 */
num_opnd (opcode)
int	opcode;
{

	switch (opcode) {

	    case CVTDF:
	    case ADDD2:
	    case SUBD2:
	    case MULD2:
	    case DIVD2:
	    case ADDF2:
	    case SUBF2:
	    case MULF2:
	    case DIVF2:	return (2);
	    case ADDF3:
	    case SUBF3:
	    case MULF3:
	    case DIVF3:
	    case ADDD3:
	    case SUBD3:
	    case MULD3:
	    case DIVD3:	return (3);
	    default:
		err ("cannot determine number of operands (opcode = %x)"
			, opcode);
		return (-1);
	}
}

/* opnd_addr	Given an operand specifier address, return the address
 *		 of the operand.
 *		Note: PC is incremented dynamically here as that is the way
 *		 the VAX goes through its operand specifiers. Also any
 *		 increment/decrement modes cause changes according to the
 *		 documentation.
 */
char *opnd_addr(type)
int	type;
{
	register unsigned temp;
	register char *tptr;
	register int mode;
	register int reg;
	extern ANYTYPE *fetch();
	extern char *addr_reg();
	extern char *pc;

	temp = (unsigned )fetch()->byte;
	mode = (temp & MODE_MASK) >> 4;
	reg = temp & REG_MASK;
	pc += 1;
	switch (mode) {
	    case INDEXED:
		tptr = (char *)(reg_cont (reg) * size_of (type));
		return (tptr + (int )opnd_addr(LONG));
	    case GENERAL:
		return (addr_reg(reg));
	    case REG_DEFRD:
		return ((char *)reg_cont (reg));
	    case AUTO_DEC:
		tptr = addr_reg(reg);
		*tptr -= size_of (type);
		return ((char *)reg_cont(reg));
	    case AUTO_INC:
		tptr = addr_reg(reg);
		*tptr += size_of (type);
		return ((char *)(reg_cont(reg) - size_of(type)));
	    case AUTO_INC_DEF:
		tptr = addr_reg (reg);
		*tptr += size_of (LONG);
		return ((char *)*(char *)(reg_cont(reg) - size_of(LONG)));
	    case BYTE_DISP:
		tptr = (char *)fetch()->byte;
		pc += 1;
		return (tptr + reg_cont(reg));
	    case BYTE_DISP_DEF:
		tptr = (char *)fetch()->byte;
		pc += 1;
		return ((char *)*(long *)(tptr + reg_cont(reg)));
	    case WORD_DISP:
		tptr = (char *)fetch()->word;
		pc += 2;
		return (tptr + reg_cont(reg));
	    case WORD_DISP_DEF:
		tptr = (char *)fetch()->word;
		pc += 2;
		return ((char *)*(long *)(tptr + reg_cont(reg)));
	    case LONG_DISP:
		tptr = (char *)fetch()->llong;
		pc += 4;
		return (tptr + reg_cont(reg));
	    case LONG_DISP_DEF:
		tptr = (char *)fetch()->llong;
		pc += 4;
		return ((char *)*(long *)(tptr + reg_cont(reg)));
	    default:				/* Cannot screw with literals */
		err ("cannot get address of operand for type 0X%x", type);
		return (NULL);
	}
}

/* get_type:	Given an opcode and the number of the operand return the data
 *		 type of the operand.
 */
get_type (opcode, opernd_num)
int	opcode;
int	opernd_num;
{

	switch (opcode) {

	    case CVTDF:	switch (opernd_num) {
			     case 1:	return (DOUBLE);
			     case 2:	return (FLOAT);
			     default:	return (GARBAGE);
			}
	    case ADDF2:
	    case SUBF2:
	    case MULF2:
	    case DIVF2:	switch (opernd_num) {
			    case 1:
			    case 2:	return (FLOAT);
			    default:	return (GARBAGE);
			}
	    case ADDF3:
	    case SUBF3:
	    case MULF3:
	    case DIVF3:	switch (opernd_num) {
			    case 1:
			    case 2:
			    case 3:	return (FLOAT);
			    default:	return (GARBAGE);
			}
	    case ADDD2:
	    case SUBD2:
	    case MULD2:
	    case DIVD2:	switch (opernd_num) {
			    case 1:
			    case 2:	return (DOUBLE);
			    default:	return (GARBAGE);
			}
	    case ADDD3:
	    case SUBD3:
	    case MULD3:
	    case DIVD3:	switch (opernd_num) {
			    case 1:
			    case 2:
			    case 3:	return (DOUBLE);
			    default:	return (GARBAGE);
			}
	    default:	return (GARBAGE);
	}
}

/* fetch:	Make a fetch from users space
 * Returned data is static. Any changes are reflected in the actual code.
 * BEWARE
 */
ANYTYPE *fetch ()
{
	extern char *pc;

	return ((ANYTYPE *)pc);
}

/* reg_cont	Return the contents of the specified register
 */
reg_cont (reg)
int	reg;
{
	if (reg == PC) return ((int )pc);
	if (reg == SP) return ((int ) &regs0t6[6]);
	if (reg == FP) return (regs0t6[-2]);
	if (reg == AP) return (regs0t6[-3]);
	if (reg >= 0 && reg <= 6) return (regs0t6[reg]);
	if (reg >= 7 && reg <= 11) return (regs7t11[reg]);
	err ("Cannot deliver contents of register %d", reg);
	return (NULL);
}

/* addr_reg	Return the stack address of the specified register
 */
char *addr_reg (reg)
int	reg;
{
	extern long *regs0t6, *regs7t11;

	if (reg < 7)
		return ((char *)(regs0t6 + reg));
	return ((char *)(regs7t11 + reg - 7));
}

/* load		Load value into the address(es) pointed to by addr.
 */
load (addr, type, valuel, valuer)
char	*addr;
int	type;
long	valuel;
long	valuer;
{

	*(long *)addr = valuel;
	if (type != QUAD)
		return;
	/* Must fix other half */
	addr = addr == addr_reg(6) ? addr_reg(7) : addr + 4;
	*(long *)addr = valuer;
}

/* err
 * The fault handler is confused. Give the user a message and continue.
 * Probably to nausea, but we might have done the job.
 */
/* VARARGS 1 */
err (fmt, a, b, c, d, e, f, g, h, i, j, k, l)
char	*fmt;
{

	fprintf (stderr, "Fault handler: ");
	fprintf (stderr, fmt, a, b, c, d, e, f, g, h, i, j, k, l);
	putc ('\n', stderr);
	fflush (stderr);		/* Deliver message NOW! */
}
