/* mt.c: Tape interface for scsi driver */
/*
 * I'm not sure I like the solution to the problem of when to
 * call scsi on the close.  If I don't sleep on mt_empty, I could
 * ask scsi to write EOFs and rewind while I still have a lot
 * of stuff on the queue.  This solution is to have mtstart()
 * wakeup mt_empty when he is called with an empty queue.
 * I sleep on mt_empty in the close routines.  The while is
 * just paranoia.
 *
 * A better solution would be to generalize the queue so that the
 * work of close could be queued and the calling process could return.
 * The open in that case would have to queue a request to open
 * and open would have to sleeep on that.
 */

#include "param.h"
#include "buf.h"
#include "dir.h"
#include "conf.h"
#include "signal.h"
#include "seg.h"
#include "ipm.h"
#include "sid.h"
#include "user.h"
#include "errno.h"

#define	b_actf	av_forw
#define	b_actl	av_back
#define	b_active b_bcount
#define	b_errcnt b_resid

#define	get(x)	((x)[0] << 24 | \
		 (x)[1] << 16 | \
		 (x)[2] << 8 | \
		 (x)[3])
#define	set(x, y)	{ (x)[2] = (y) >> 16 & 0xff; \
			  (x)[3] = (y) >> 8 & 0xff; \
			  (x)[4] = (y) & 0xff; }

#define	SEEK		1
#define	SIO		2
#define RS		3
#define	VIPER		5
#define	MAXERRORS	1
#define	ST_WRITTEN	02


static	char read_data[12] =	{ 0x08, 1, 0, 0, 0, 0 };
static	char write_data[12] = 	{ 0x0A, 1, 0, 0, 0, 0 };
static	char request_sense[12] ={ 0x03, 0, 0, 0, 14, 0 };
static	char unit_ready[12] = 	{ 0x00, 0, 0, 0, 0, 0 };
static	char seek[12] =		{ 0x0C, 0, 0, 0, 0, 0 };
static	char rewind[12] = 	{ 0x01, 1, 0, 0, 0, 0 };
static	char release[12] = 	{ 0x17, 0, 0, 0, 0, 0 };
static	char write_eof[12] = 	{ 0x10, 0, 0, 0, 2, 0 };
static	char backup_1[12] = 	{ 0x11, 1, 0xff, 0xff, 0xff, 0 };
static	struct buf rmtbuf;		/* for raw i/o */

static	char rs_data[14];
static	char mode_data[512];
static	char *cdb;
struct buf mttab;		/* io queue for block device */
static	int tape_flag;
static	int mt_empty = 1;	/* true if queue is empty */

mtopen(dev, flag)
{
	int 		mtopenintr();
	register	unit;
	int		scsi_status;
	int		openable;

	if (tape_flag) {
		u.u_error = ENXIO;
		return;
	}

	openable = 0;
	unit = minor(dev) & 07;
	scsi_status = scsi_unit_ready(unit);
	scsi_status = scsi_req_sense(unit, rs_data, sizeof(rs_data));
	if (scsi_status == 0) {
		switch (rs_data[2] & 0x0f) {
		case 0x00:
		case 0x01:
			openable = 1;
			break;
			
		case 0x06:
			scsi_status = scsi_unit_ready(unit);
			openable = (scsi_status == 0) ? 1 : 0;
			break;
			
		default:
			printf("mtopen: status = 0x%x sense = 0x%x\n",
			       scsi_status, rs_data[2]);
			break;
		}
	}
	if (!openable) {
		u.u_error = ENXIO;
		return;
	}
	mttab.b_flags |= B_TAPE;
	tape_flag = 1;
	return;
}

mtclose(dev, flag)
{
	int mtopenintr();
	register unit;

	unit = minor(dev) & 07;
	splbio();
	while (mt_empty == 0)
		sleep(&mt_empty, PRIBIO);
	spl0();
	if (tape_flag & ST_WRITTEN) {
		scsi(minor(dev), B_WRITE, write_eof, 0, 0, mtopenintr);
		sleep(&tape_flag, PRIBIO);
		scsi(minor(dev), B_WRITE, backup_1, 0, 0, mtopenintr);
		sleep(&tape_flag, PRIBIO);
	}
	if ((minor(dev) & 0200) == 0) {	/* rewind */
		scsi(unit, B_WRITE, rewind, 0, 0, mtopenintr);
		sleep(&tape_flag, PRIBIO);
	}
	tape_flag = 0;
}

static
mtopenintr(status, resid)
int	status;
long	resid;
{
	wakeup(&tape_flag);
}

mtstrategy(bp)
	register struct buf *bp;
{
	bp->av_forw = NULL;
	splbio();
	bp->b_errcnt = 0;
	if (mttab.b_actf == NULL) {
		mttab.b_actf = bp;
		mt_empty = 0;
	} else
		mttab.b_actl->av_forw = bp;
	mttab.b_actl = bp;
	if (mttab.b_active == 0)
		mtstart();
	spl0();
}

mtread(dev)
{
	physio(mtstrategy, &rmtbuf, dev, B_READ);
}


mtwrite(dev)
{
	physio(mtstrategy, &rmtbuf, dev, B_WRITE);
}

static
mtstart()
{
	int mtintr();
	register struct buf *bp;

	if ((bp = mttab.b_actf) == NULL) {
		mt_empty = 1;
		mttab.b_active = 0;
		wakeup(&mt_empty);
		return;
	}
	mttab.b_active = SIO;
	cdb = (bp->b_flags & B_READ) ? read_data : write_data;
	set(cdb, bp->b_bcount + 511 >> 9);
	if ((bp->b_flags & B_READ) == 0)
		tape_flag |= ST_WRITTEN;
	scsi(minor(bp->b_dev)&07, bp->b_flags&B_READ,
		cdb, bp->b_un.b_addr, bp->b_bcount + 511 & ~0777, mtintr);
}

static
mtintr(status, resid)
int	status;
long	resid;
{
	register struct buf *bp;
	int	i;


	if (status != 0 || resid != 0) {
		printf("mtintr: status = 0x%x resid = 0x%x\n", status, resid);
	}
	if ((bp = mttab.b_actf) == NULL)
		return;
	switch (mttab.b_active) {
	case RS:
		if (rs_data[2] & 0x80) { 	/* EOF */
			bp->b_resid = bp->b_bcount;
			break;
		}
		printf("mtintr: rs_data[2] = 0x%x\n", rs_data[2]);
		bp->b_flags |= B_ERROR;
		break;
	default:
		if (status & 2) {		/* check condition */
			mttab.b_active = RS;
			scsi(minor(bp->b_dev)&07, B_READ,
			     request_sense, rs_data, sizeof(rs_data),
			     mtintr);
			return;
		}
		break;
	}
	mttab.b_actf = bp->av_forw;
	bp->b_resid = resid;
	iodone(bp);
	mtstart();
}
