
/*
 * $Log:	process.c,v $
 * Revision 1.1  86/03/11  10:07:28  root
 * Initial revision
 * 
 */

/* Copyright (c) 1982 Regents of the University of California */


/*
 * Process management.
 *
 * This module contains the routines to manage the execution and
 * tracing of the debuggee process.
 */

#include "defs.h"
#include "process.h"
#include "machine.h"
#include "events.h"
#include "tree.h"
#include "eval.h"
#include "operators.h"
#include "source.h"
#include "object.h"
#include "mappings.h"
#include "main.h"
#include "coredump.h"
#include <signal.h>
#include <errno.h>
#include <sys/param.h>
#include <sys/dir.h>
#include <sys/seg.h>
#include <sys/ipm.h>
#include <sys/user.h>
#include <sys/reg.h>
#include <sys/stat.h>
#include <sys/wait.h>

#ifndef public

typedef struct Process *Process;

Process process;

#define DEFSIG -1

#include "machine.h"

#endif public

#define NOTSTARTED 1
#define STOPPED 0177
#define FINISHED 0
# define SIGTRAP_KLUDGE		/* dbx can't deal with SIGTRAP handlers */

/*
 * A cache of the instruction segment is kept to reduce the number
 * of system calls.  Might be better just to read the entire
 * code space into memory.
 */

#define CSIZE 1003       /* size of instruction cache */

typedef struct {
    Word addr;
    Word val;
} CacheWord;

/*
 * This structure holds the information we need from the user structure.
 */

struct Process {
    int pid;			/* process being traced */
    int mask;			/* process status word */
    Word reg[NREG];		/* process' registers */
    Word oreg[NREG];		/* registers when process last stopped */
    short status;		/* either STOPPED or FINISHED */
    short signo;		/* signal that stopped process */
    int exitval;		/* return value from exit() */
    long sigset;		/* bit array of traced signals */
    CacheWord word[CSIZE];	/* text segment cache */
    Ttyinfo ttyinfo;		/* process' terminal characteristics */
    Address sigstatus;		/* process' handler for current signal */
};

/*
 * These definitions are for the arguments to "pio".
 */

typedef enum { PREAD, PWRITE } PioOp;
typedef enum { DTEXTSEG, DDATASEG } PioSeg;

private struct Process pbuf;

#define MAXNCMDARGS 100         /* maximum number of arguments to RUN */

extern int errno;

private Boolean just_started;
private int argc;
private String argv[MAXNCMDARGS];
private String infile, outfile;

/*
 * Initialize process information.
 */

public process_init()
{
    register Integer i;
    Char buf[10];

    process = &pbuf;
    process->status = (coredump) ? STOPPED : NOTSTARTED;
    setsigtrace();
# ifdef mc68000
    for (i = 0; i < NREG/2; i++) {
	sprintf(buf, "$d%d", i);
	defregname(identname(buf, false), i);
	sprintf(buf, "$a%d", i);
	defregname(identname(buf, false), i+8);
    }
# else (vax)
    for (i = 0; i < NREG; i++) {
	sprintf(buf, "$r%d", i);
	defregname(identname(buf, false), i);
    }
    defregname(identname("$ap", true), ARGP);
# endif mc68000
    defregname(identname("$fp", true), FRP);
    defregname(identname("$sp", true), STKP);
    defregname(identname("$pc", true), PROGCTR);
    if (coredump) {
	coredump_readin(process->mask, process->reg, process->signo);
	pc = process->reg[PROGCTR];
	getsrcpos();
    }
    arginit();
}

/*
 * Routines to get at process information from outside this module.
 */

public Word reg(n)
Integer n;
{
    register Word w;

    if (n == NREG) {
	w = process->mask;
    } else {
	w = process->reg[n];
    }
    return w;
}

public setreg(n, w)
Integer n;
Word w;
{
    process->reg[n] = w;
}

/*
 * Begin execution.
 *
 * We set a breakpoint at the end of the code so that the
 * process data doesn't disappear after the program terminates.
 */

private Boolean remade();

start(argv, infile, outfile)
String argv[];
String infile, outfile;
{
    String pargv[4];
    Node cond;

    if (coredump) {
	coredump = false;
	fclose(corefile);
	coredump_close();
    }
    if (argv == nil) {
	argv = pargv;
	pargv[0] = objname;
	pargv[1] = nil;
    } else {
	argv[argc] = nil;
    }
    if (remade(objname)) {
	reinit(argv, infile, outfile);
    }
    pstart(process, argv, infile, outfile);
    if (process->status == STOPPED) {
	pc = 0;
	setcurfunc(program);
	if (objsize != 0) {
	    cond = build(O_EQ, build(O_SYM, pcsym), build(O_LCON, lastaddr()));
	    event_once(cond, buildcmdlist(build(O_ENDX)));
	}
    }
}

/*
 * Check to see if the object file has changed since the symbolic
 * information last was read.
 */

private time_t modtime;

private Boolean remade(filename)
String filename;
{
    struct stat s;
    Boolean b;

    stat(filename, &s);
    b = (Boolean) (modtime != 0 and modtime < s.st_mtime);
    modtime = s.st_mtime;
    return b;
}

/*
 * Set up what signals we want to trace.
 */

private setsigtrace()
{
    register Integer i;
    register Process p;

    p = process;
    for (i = 1; i <= NSIG; i++) {
	psigtrace(p, i, true);
    }
    psigtrace(p, SIGHUP, false);
    psigtrace(p, SIGKILL, false);
    psigtrace(p, SIGALRM, false);
#ifdef notdef
	/* gun doesn't have these */
    psigtrace(p, SIGTSTP, false);
    psigtrace(p, SIGCONT, false);
    psigtrace(p, SIGCHLD, false);
#endif
}

/*
 * Initialize the argument list.
 */

public arginit()
{
    infile = nil;
    outfile = nil;
    argv[0] = objname;
    argc = 1;
}

/*
 * Add an argument to the list for the debuggee.
 */

public newarg(arg)
String arg;
{
    if (argc >= MAXNCMDARGS) {
	error("too many arguments");
    }
    argv[argc++] = arg;
}

/*
 * Set the standard input for the debuggee.
 */

public inarg(filename)
String filename;
{
    if (infile != nil) {
	error("multiple input redirects");
    }
    infile = filename;
}

/*
 * Set the standard output for the debuggee.
 * Probably should check to avoid overwriting an existing file.
 */

public outarg(filename)
String filename;
{
    if (outfile != nil) {
	error("multiple output redirect");
    }
    outfile = filename;
}

/*
 * Start debuggee executing.
 */

public run()
{
    process->status = STOPPED;
    fixbps();
    curline = 0;
    start(argv, infile, outfile);
    just_started = true;
    isstopped = false;
    cont(0);
}

/*
 * Continue execution wherever we left off.
 *
 * Note that this routine never returns.  Eventually bpact() will fail
 * and we'll call printstatus or step will call it.
 */

typedef int Intfunc();

private Intfunc *dbintr;
private intr();

#define succeeds    == true
#define fails       == false

public cont(signo)
integer signo;
{
    integer s;

    dbintr = signal(SIGINT, intr);
    if (just_started) {
	just_started = false;
    } else {
	if (not isstopped) {
	    error("can't continue execution");
	}
	isstopped = false;
	stepover();
    }
    s = signo;
    for (;;) {
	if (single_stepping) 
	   {
	   printnews();
           if (traceexec) { printf("cont() single stepping\n"); fflush(stdout); }
	   } 
	else 
	   {
    	   if (traceexec) { printf("cont() ->resume()\n"); fflush(stdout); }
	   setallbps();
	   resume(s);
	   unsetallbps();
	   s = DEFSIG;
	   if (bpact() fails) 
		printstatus();
	   }
	stepover();
        }
    /* NOTREACHED */
}

/*
 * This routine is called if we get an interrupt while "running" px
 * but actually in the debugger.  Could happen, for example, while
 * processing breakpoints.
 *
 * We basically just want to keep going; the assumption is
 * that when the process resumes it will get the interrupt
 * which will then be handled.
 */

private intr()
{
    signal(SIGINT, intr);
}

public fixintr()
{
    signal(SIGINT, dbintr);
}

/*
 * Resume execution.
 */

public resume(signo)
int signo;
{
    register Process p;
int exitcode;


    p = process;
    pcont(p, signo);
    pc = process->reg[PROGCTR];
    if (p->status != STOPPED) {
	if (p->signo != 0) {
	    error("program terminated by signal %d", p->signo);
	} else if (not runfirst) {
	    error("program unexpectely exited with signal %d", p->exitval);
	}
    }
}

/*
 * Continue execution up to the next source line.
 *
 * There are two ways to define the next source line depending on what
 * is desired when a procedure or function call is encountered.  Step
 * stops at the beginning of the procedure or call; next skips over it.
 */

/*
 * Stepc is what is called when the step command is given.
 * It has to play with the "isstopped" information.
 */

public stepc()
{
    if (not isstopped) {
	error("can't continue execution");
    }
    isstopped = false;
    dostep(false);
    isstopped = true;
}

public next()
{
    Address oldfrp, newfrp;

    if (not isstopped) {
	error("can't continue execution");
    }
    isstopped = false;
    oldfrp = reg(FRP);
    do {
	dostep(true);
	pc = reg(PROGCTR);
	newfrp = reg(FRP);
    } while (newfrp < oldfrp and newfrp != 0);
    isstopped = true;
}

/*
 * Continue execution until the current function returns, or,
 * if the given argument is non-nil, until execution returns to
 * somewhere within the given function.
 */

public rtnfunc (f)
Symbol f;
{
    Address addr;
    Symbol t;

    if (not isstopped) {
	error("can't continue execution");
    } else if (f != nil and not isactive(f)) {
	error("%s is not active", symname(f));
    } else {
	addr = return_addr();
	if (addr == nil) {
	    error("no place to return to");
	} else {
	    isstopped = false;
	    contto(addr);
	    if (f != nil) {
		for (;;) {
		    t = whatblock(pc);
		    addr = return_addr();
		if (t == f or addr == nil) break;
		    contto(addr);
		}
	    }
	    if (bpact() fails) {
		isstopped = true;
		printstatus();
	    }
	}
    }
}

/*
 * Single-step over the current machine instruction.
 *
 * If we're single-stepping by source line we want to step to the
 * next source line.  Otherwise we're going to continue so there's
 * no reason to do all the work necessary to single-step to the next
 * source line.
 */

public stepover()
{
    Boolean b;

    if (traceexec) {
	printf("!! stepping over 0x%x\n", process->reg[PROGCTR]); 
	fflush(stdout);
    }
    if (single_stepping) {
    if (traceexec) {
	printf("!! single stepping loop 0x%x\n", process->reg[PROGCTR]);
	fflush(stdout);
    }
	dostep(false);
    } else {
    if (traceexec) {
	printf("Instruction tracing to dostep()\n");
	fflush(stdout);
    }
	b = inst_tracing;
	inst_tracing = true;
	dostep(false);
	inst_tracing = b;
    }
    if (traceexec) {
	printf("!! stepped to 0x%x\n", process->reg[PROGCTR]);
	fflush(stdout);
    }
}

/*
 * Resume execution up to the given address.  It is assumed that
 * no breakpoints exist between the current address and the one
 * we're stepping to.  This saves us from setting all the breakpoints.
 */

public stepto(addr)
Address addr;
{
    xto(addr, false);
}

public contto (addr)
Address addr;
{
    xto(addr, true);
}

public xto (addr, catchbps)
Address addr;
boolean catchbps;
{
    Address curpc;

    if (catchbps) {
	if (traceexec) {
	    printf("xto()  -> stepover()\n");
	    fflush(stdout);
	}
	stepover();
    }
    curpc = process->reg[PROGCTR];

    if (addr != curpc) {
	if (traceexec) {
	    printf("xto()  stepping from 0x%x to 0x%x\n", curpc, addr);
	    fflush(stdout);
	}
	if (catchbps) {
	    setallbps();
	}
	setbp(addr);
	resume(DEFSIG);
	unsetbp(addr);

	
	if (catchbps) {
	    unsetallbps();
	 }
	if (not isbperr()) {
	    printstatus();
	}
    }
}

/*
 * Print the status of the process.
 * This routine does not return.
 */

public printstatus()
{
    int status;

    if (process->status == FINISHED) {
	exit(0);
    } else {
	setcurfunc(whatblock(pc));
	getsrcpos();
	if (process->signo == SIGINT) {
	    isstopped = true;
	    printerror();
	} else if (isbperr() and isstopped) {
	    printf("stopped ");
	    printloc();
	    putchar('\n');
	    if (curline > 0) {
		printlines(curline, curline);
	    } else {
		printinst(pc, pc);
	    }
	    erecover();
	} else {
	    fixintr();
	    isstopped = true;
	    printerror();
	}
    }
}

/*
 * Print out the current location in the debuggee.
 */

public printloc()
{
    printf("in ");
    printname(stdout, curfunc);
    putchar(' ');
    if (curline > 0 and not useInstLoc) {
	printsrcpos();
    } else {
	useInstLoc = false;
	curline = 0;
	printf("at 0x%x", pc);
    }
}

/*
 * Some functions for testing the state of the process.
 */

public Boolean notstarted(p)
Process p;
{
    return (Boolean) (p->status == NOTSTARTED);
}

public Boolean isfinished(p)
Process p;
{
    return (Boolean) (p->status == FINISHED);
}

/*
 * Return the signal number which stopped the process.
 */

public Integer errnum(p)
Process p;
{
    return p->signo;
}

/*
 * Return the termination code of the process.
 */

public Integer exitcode(p)
Process p;
{
    return p->exitval;
}

/*
 * These routines are used to access the debuggee process from
 * outside this module.
 *
 * They invoke "pio" which eventually leads to a call to "ptrace".
 * The system generates an I/O error when a ptrace fails.  During reads
 * these are ignored, during writes they are reported as an error, and
 * for anything else they cause a fatal error.
 */

extern Intfunc *onsyserr();

private badaddr;
private read_err(), write_err();

/*
 * Read from the process' instruction area.
 */

public iread(buff, addr, nbytes)
char *buff;
Address addr;
int nbytes;
{
    Intfunc *f;

    f = onsyserr(EIO, read_err);
    badaddr = addr;
    if (coredump) {
	coredump_readtext(buff, addr, nbytes);
    } else {
	pio(process, PREAD, DTEXTSEG, buff, addr, nbytes);
    }
    onsyserr(EIO, f);
}

/* 
 * Write to the process' instruction area, usually in order to set
 * or unset a breakpoint.
 */

public iwrite(buff, addr, nbytes)
char *buff;
Address addr;
int nbytes;
{
    Intfunc *f;

    if (coredump) {
	error("no process to write to");
    }
    f = onsyserr(EIO, write_err);
    badaddr = addr;
    pio(process, PWRITE, DTEXTSEG, buff, addr, nbytes);
    onsyserr(EIO, f);
}

/*
 * Read for the process' data area.
 */

public dread(buff, addr, nbytes)
char *buff;
Address addr;
int nbytes;
{
    Intfunc *f;

    f = onsyserr(EIO, read_err);
    badaddr = addr;
    if (coredump) {
	coredump_readdata(buff, addr, nbytes);
      if (frametrace){printf("addr=%X nbytes=%X *buff=%X\n",addr,nbytes,*(int *)buff); }
    } else {
	pio(process, PREAD, DDATASEG, buff, addr, nbytes);
    }
    onsyserr(EIO, f);
}

/*
 * Write to the process' data area.
 */

public dwrite(buff, addr, nbytes)
char *buff;
Address addr;
int nbytes;
{
    Intfunc *f;

    if (coredump) {
	error("no process to write to");
    }
    f = onsyserr(EIO, write_err);
    badaddr = addr;
    pio(process, PWRITE, DDATASEG, buff, addr, nbytes);
    onsyserr(EIO, f);
}

/*
 * Trap for errors in reading or writing to a process.
 * The current approach is to "ignore" read errors and complain
 * bitterly about write errors.
 */

private read_err()
{
    /*
     * Ignore.
     */
}

private write_err()
{
    if (coredump)
	{
	printf("Cannot execute core file!!\n");
	fflush(stdout);
	}
    else
    printf("Child process may be busy!!\n");fflush(stdout);
    error("can't write to process (address 0x%x) errcode = %d", badaddr,errno);
}

/*
 * Ptrace interface.
 */

/*
 * This magic macro enables us to look at the process' registers
 * in its user structure.
 */
# ifndef vax
struct user u;
#define regloc(reg)     (long)&u.u_ar0[(reg)]
# else (vax)
#define regloc(reg)     (ctob(UPAGES) + ( sizeof(int) * (reg) ))
# endif vax
#define WMASK           (~(sizeof(Word) - 1))


#define cachehash(addr) ((unsigned) ((addr >> 2) % CSIZE))

#define FIRSTSIG        SIGINT
#define LASTSIG         SIGQUIT
#define ischild(pid)    ((pid) == 0)
#define traceme()       myptrace(0, 0, 0, 0)
#define setrep(n)       (1 << ((n)-1))
#define istraced(p)     (p->sigset&setrep(p->signo))

/*
 * Ptrace options (specified in first argument).
 */

#define UREAD   3       /* read from process's user structure */
#define UWRITE  6       /* write to process's user structure */
#define IREAD   1       /* read from process's instruction space */
#define IWRITE  4       /* write to process's instruction space */
#define DREAD   2       /* read from process's data space */
#define DWRITE  5       /* write to process's data space */
#define CONT    7       /* continue stopped process */
#define SSTEP   9       /* continue for approximately one instruction */
#define PKILL   8       /* terminate the process */
#ifdef tws
#define RREAD   10      /* read from process's registers */
#define RWRITE  11      /* write to process's registers */
#endif tws

# ifdef MC68K
nullsig()
{
}
# endif

/*
 * Start up a new process by forking and exec-ing the
 * given argument list, returning when the process is loaded
 * and ready to execute.  The PROCESS information (pointed to
 * by the first argument) is appropriately filled.
 *
 * If the given PROCESS structure is associated with an already running
 * process, we terminate it.
 */

/* VARARGS2 */
pstart(p, argv, infile, outfile)
Process p;
String argv[];
String infile;
String outfile;
{
    int status;

    if (p->pid != 0) {
	pterm(p);
    }
    psigtrace(p, SIGTRAP, true);
    p->pid = fork();
    if (p->pid == -1) {
	panic("can't fork");
    }
    if (ischild(p->pid)) {
# ifdef MC68K
    	signal(SIGTRAP,nullsig);
# endif
	traceme();
	if (infile != nil) {
	    infrom(infile);
	}
	if (outfile != nil) {
	    outto(outfile);
	}
	execv(argv[0], argv);
	write(2, "can't exec ", 11);
	write(2, argv[0], strlen(argv[0]));
	write(2, "\n", 1);
	_exit(1);
    }
    pwait(p->pid, &status);
    getinfo(p, status);
    if (p->status != STOPPED) {
	error("program could not begin execution");
    }
    ptraced(p->pid);
}

/*
 * Terminate a ptrace'd process.
 */

public pterm (p)
Process p;
{
    integer status;

    if (p != nil and p->pid != 0) {
	myptrace(PKILL, p->pid, 0, 0);
	pwait(p->pid, &status);
	unptraced(p->pid);
    }
}

/*
 * Continue a stopped process.  The first argument points to a Process
 * structure.  Before the process is restarted it's user area is modified
 * according to the values in the structure.  When this routine finishes,
 * the structure has the new values from the process's user area.
 *
 * Pcont terminates when the process stops with a signal pending that
 * is being traced (via psigtrace), or when the process terminates.
 */

private pcont(p, signo)
Process p;
int signo;
{
    int s;
     int status;
    register Address pc;

    if (p->pid == 0) {
	error("program not active");
    }
    s = signo;
    do {
	setinfo(p, s);
	if (traceexec) {
	    printf("!! pcont from 0x%x with signal %d (%d)\n",
		p->reg[PROGCTR], s, p->signo);
	    fflush(stdout);
	}
	sigs_off();
	if (myptrace(CONT, p->pid, p->reg[PROGCTR], p->signo) < 0) {
	    panic("error %d trying to continue process", errno);
	}
	if (traceexec) {
	    printf("!! pcont ->pwait()from \n");
	    fflush(stdout);
	}
	pwait(p->pid, &status);
	sigs_on();
	if (traceexec) {
	    printf("!! pcont -> getinfo() \n");
	    fflush(stdout);
	}
	getinfo(p, status);
	if (traceexec and not istraced(p)) {
	    printf("!! ignored signal %d at 0x%x\n", p->signo, p->reg[PROGCTR]);
	    fflush(stdout);
	}
	s = p->signo;
    } while (p->status == STOPPED and not istraced(p));

#ifdef mc68000
    /* on the 68000, breakpoints are implemented via the TRAP instructions;
     * this leaves the pc pointing to the next instruction,
     * correct for that here.
     */
/* This is not correct when the correct TRAP (0xf) is used.. The pc is being 
   reset elsewhere..

    if (p->signo = SIGTRAP)
	p->reg[PROGCTR] -= 2;
*/

#endif mc68000
    if (traceexec) {
	printf("!!pcont() stopped at 0x%x on signal %d\n", p->reg[PROGCTR], p->signo);
	fflush(stdout);
    }
}

/*
 * Single step as best ptrace can.
 */

public pstep(p, signo)
Process p;
integer signo;
{
    int status;

    setinfo(p, signo);
    if (traceexec) {
	printf("!! pstep from pc 0x%x with signal %d (%d)\n",
	    p->reg[PROGCTR], signo, p->signo);
	fflush(stdout);
    }
    sigs_off();
    if (myptrace(SSTEP, p->pid, p->reg[PROGCTR], p->signo) < 0) {
	panic("error %d trying to step process", errno);
    }
    pwait(p->pid, &status);
    sigs_on();
    getinfo(p, status);
    if (traceexec) {
	printf("!! pstep to pc 0x%x on signal %d\n", p->reg[PROGCTR], p->signo);
	fflush(stdout);
    }
    if (p->status != STOPPED) {
	error("program unexpectedly exited with %d\n", p->exitval);
    }
}

/*
 * Return from execution when the given signal is pending.
 */

public psigtrace(p, sig, sw)
Process p;
int sig;
Boolean sw;
{
    if (sw) {
	p->sigset |= setrep(sig);
    } else {
	p->sigset &= ~setrep(sig);
    }
}

/*
 * Don't catch any signals.
 * Particularly useful when letting a process finish uninhibited.
 */

public unsetsigtraces(p)
Process p;
{
    p->sigset = 0;
}

/*
 * Turn off attention to signals not being caught.
 */

private Intfunc *sigfunc[NSIG];

private sigs_off()
{
    register int i;

    for (i = FIRSTSIG; i < LASTSIG; i++) {
	if (i != SIGKILL) {
	    sigfunc[i] = signal(i, SIG_IGN);
	}
    }
}

/*
 * Turn back on attention to signals.
 */

private sigs_on()
{
    register int i;

    for (i = FIRSTSIG; i < LASTSIG; i++) {
	if (i != SIGKILL) {
	    signal(i, sigfunc[i]);
	}
    }
}

/*
 * Get process information from user area.
 */

private int rloc[] ={
# ifdef tws
    RD0, RD1, RD2, RD3, RD4, RD5, RD6, RD7,
    RA0, RA1, RA2, RA3, RA4, RA5, RA6, USP,
    PC, PS
# endif tws
# ifdef mc68000
    R0, R1, R2, R3, R4, R5, R6, R7, R8, R9, R10, R11, R12, R13, FP, SP, PC
# endif mc680000
# ifdef vax
    R0, R1, R2, R3, R4, R5, R6, R7, R8, R9, R10, R11, AP, FP, SP, PC
# endif vax
};

private getinfo(p, status)
register Process p;
int status;
{
    register int i;
    Address addr;

    p->signo = status & 0xff;
    p->exitval = (status >> 8) & 0xff;
    if (p->signo != STOPPED) {
	p->status = FINISHED;
	p->pid = 0;
	p->reg[PROGCTR] = 0;
    } else {
	p->status = p->signo;
	p->signo = p->exitval;
	p->exitval = 0;
# ifdef MC68K
	u.u_ar0 = (int *) (myptrace(UREAD, p->pid,
				&((struct user *)0)->u_ar0, 0)
			& ((UPAGES * NBPG) -1));
# endif MC68K
#ifdef tws
	p->mask = myptrace(RREAD, p->pid, PS, 0);
#else
	p->mask = myptrace(UREAD, p->pid, regloc(PS), 0);
#endif tws
	for (i = 0; i < NREG; i++) {
#ifdef tws
	    p->reg[i] = myptrace(RREAD, p->pid, rloc[i], 0);
#else
	    p->reg[i] = myptrace(UREAD, p->pid, regloc(rloc[i]), 0);
#endif tws
	    p->oreg[i] = p->reg[i];
	}
	savetty(stdout, &(p->ttyinfo));
	addr = (Address) &(((struct user *) 0)->u_signal[p->signo]);
	p->sigstatus = (Address) myptrace(UREAD, p->pid, addr, 0);
    }
}

/*
 * Set process's user area information from given process structure.
 */

private setinfo(p, signo)
register Process p;
int signo;
{
    register int i;
    register int r;

# ifdef SIGTRAP_KLUDGE
    if (p->signo == SIGTRAP && p->sigstatus != 0 && p->sigstatus != 1) {
	p->signo = 0;
	p->sigstatus = 0;
    }
# endif SIGTRAP_KLUDGE

    if (signo == DEFSIG) {
	if (istraced(p) and (p->sigstatus == 0 or p->sigstatus == 1)) {
	    p->signo = 0;
	}
    } else {
	p->signo = signo;
    }
    for (i = 0; i < NREG; i++) {
	if ((r = p->reg[i]) != p->oreg[i]) {
#ifdef tws
	    myptrace(RWRITE, p->pid, rloc[i], r);
#else
	    myptrace(UWRITE, p->pid, regloc(rloc[i]), r);
#endif tws
	}
    }
    restoretty(stdout, &(p->ttyinfo));
}

/*
 * Return the address associated with the current signal.
 */

public Address usignal (p)
Process p;
{
    Address r;

# ifdef SIGTRAP_KLUDGE 			/* see above */
    if (p->signo == SIGTRAP && p->sigstatus != 0 && p->sigstatus != 1)
	return 0;
# endif SIGTRAP_KLUDGE

    r = p->sigstatus;
#ifdef vax
    /* (Plus two since the address points to the beginning of a proc). */
    if (r != 0 and r != 1) {
	r += 2;
    }
#endif vax
    return r;
}

/*
 * Structure for reading and writing by words, but dealing with bytes.
 */

typedef union {
    Word pword;
    Byte pbyte[sizeof(Word)];
} Pword;

/*
 * Read (write) from (to) the process' address space.
 * We must deal with ptrace's inability to look anywhere other
 * than at a word boundary.
 */

private Word fetch();
private store();

private pio(p, op, seg, buff, addr, nbytes)
Process p;
PioOp op;
PioSeg seg;
char *buff;
Address addr;
int nbytes;
{
    register int i;
    register Address newaddr;
    register char *cp;
    char *bufend;
    Pword w;
    Address wordaddr;
    int byteoff;

    if (p->status != STOPPED) {
	error("program is not active");
    }
    cp = buff;
    newaddr = addr;
    wordaddr = (newaddr&WMASK);
    if (wordaddr != newaddr) {
	w.pword = fetch(p, seg, wordaddr);
	for (i = newaddr - wordaddr; i < sizeof(Word) and nbytes > 0; i++) {
	    if (op == PREAD) {
		*cp++ = w.pbyte[i];
	    } else {
		w.pbyte[i] = *cp++;
	    }
	    nbytes--;
	}
	if (op == PWRITE) {
	    store(p, seg, wordaddr, w.pword);
	}
	newaddr = wordaddr + sizeof(Word);
    }
    byteoff = (nbytes&(~WMASK));
    nbytes -= byteoff;
    bufend = cp + nbytes;
    while (cp < bufend) {
	if (op == PREAD) {
	    *((Word *) cp) = fetch(p, seg, newaddr);
	} else {
	    store(p, seg, newaddr, *((Word *) cp));
	}
	cp += sizeof(Word);
	newaddr += sizeof(Word);
    }
    if (byteoff > 0) {
	w.pword = fetch(p, seg, newaddr);
	for (i = 0; i < byteoff; i++) {
	    if (op == PREAD) {
		*cp++ = w.pbyte[i];
	    } else {
		w.pbyte[i] = *cp++;
	    }
	}
	if (op == PWRITE) {
	    store(p, seg, newaddr, w.pword);
	}
    }
}

/*
 * Get a word from a process at the given address.
 * The address is assumed to be on a word boundary.
 *
 * A simple cache scheme is used to avoid redundant ptrace calls
 * to the instruction space since it is assumed to be pure.
 *
 * It is necessary to use a write-through scheme so that
 * breakpoints right next to each other don't interfere.
 */

private Integer nfetchs, nreads, nwrites;

private Word fetch(p, seg, addr)
Process p;
PioSeg seg;
register int addr;
{
    register CacheWord *wp;
    register Word w;

    switch (seg) {
	case DTEXTSEG:
	    ++nfetchs;
	    wp = &p->word[cachehash(addr)];
	    if (addr == 0 or wp->addr != addr) {
		++nreads;
		w = myptrace(IREAD, p->pid, addr, 0);
		wp->addr = addr;
		wp->val = w;
	    } else {
		w = wp->val;
	    }
	    break;

	case DDATASEG:
	    w = myptrace(DREAD, p->pid, addr, 0);
	    break;

	default:
	    panic("fetch: bad seg %d", seg);
	    /* NOTREACHED */
    }
    return w;
}

/*
 * Put a word into the process' address space at the given address.
 * The address is assumed to be on a word boundary.
 */

private store(p, seg, addr, data)
Process p;
PioSeg seg;
int addr;
Word data;
{
    register CacheWord *wp;

    switch (seg) {
	case DTEXTSEG:
	    ++nwrites;
	    wp = &p->word[cachehash(addr)];
	    wp->addr = addr;
	    wp->val = data;
	    myptrace(IWRITE, p->pid, addr, data);
	    break;

	case DDATASEG:
	    myptrace(DWRITE, p->pid, addr, data);
	    break;

	default:
	    panic("store: bad seg %d", seg);
	    /* NOTREACHED */
    }
}

public printptraceinfo()
{
    printf("%d fetchs, %d reads, %d writes\n", nfetchs, nreads, nwrites);
}

/*
 * Redirect input.
 * Assuming this is called from a child, we should be careful to avoid
 * (possibly) shared standard I/O buffers.
 */

private infrom (filename)
String filename;
{
    Fileid in;

    in = open(filename, 0);
    if (in == -1) {
	write(2, "can't read ", 11);
	write(2, filename, strlen(filename));
	write(2, "\n", 1);
	_exit(1);
    }
    fswap(0, in);
}

/*
 * Redirect standard output.
 * Same assumptions as for "infrom" above.
 */

private outto (filename)
String filename;
{
    Fileid out;

    out = creat(filename, 0666);
    if (out == -1) {
	write(2, "can't write ", 12);
	write(2, filename, strlen(filename));
	write(2, "\n", 1);
	_exit(1);
    }
    fswap(1, out);
}

/*
 * Swap file numbers, useful for redirecting standard input or output.
 */

private fswap(oldfd, newfd)
Fileid oldfd;
Fileid newfd;
{
    if (oldfd != newfd) {
	close(oldfd);
	dup(newfd);
	close(newfd);
    }
}
