#
/*
 *
 * UNIX debugger
 * Instruction printing routines
 * MACHINE DEPENDENT
 */

#include "defs.h"

STRING	errflg;
L_INT	dot;
INT	dotinc;
L_INT	var[];
POS	type, space, incp;


char *badop = "\t???";

char *brname[16] = {
	"ra", "sr", "hi", "ls", "cc", "cs", "ne", "eq",
	"vc", "vs", "pl", "mi", "ge", "lt", "gt", "le"
};

char *condname[16] = {
	"t",  "f",  "hi", "ls", "cc", "cs", "ne", "eq",
	"vc", "vs", "pl", "mi", "ge", "lt", "gt", "le"
};

char *shname[4] = {
	"as", "ls", "rox", "ro"
};

char *opname[4] = {
	"tst", "chg", "clr", "set"
};

int omove (), obranch (), oimmed (), oprint ();
int oneop (), soneop (), oreg (), ochk ();
int olink (), omovem (), oquick (), omoveq ();
int otrap (), oscc (), opmode (), shift ();
int extend (), biti (), oconst (), odbcc ();
int omovep (), omovec (), omoves (), mvspec ();
int olea (), odiv ();
int ocallm (), opack (), ochk2 (), bfoneop (), bftwoop ();
int ocas (), ocas2 (), otrapcc (), regpair ();
int fops (), fbcc (), fdbcc (), fscc (), ftrapcc ();
int instfetch ();
static	long xlookup ();
char *printval ();

extern char *malloc ();
extern char *strcpy ();

struct opdesc {
	unsigned mask, match;
	int (*opfun) ();
	char *farg;
} opdecode[] =
{	/* order is important below */
	/* move instructions */
	0xF000, 0x1000, omove,	"b",
	0xF000, 0x2000, omove,	"l",
	0xF000, 0x3000, omove,	"w",

	/* branches */
	0xF000, 0x6000, obranch, 0,

	/* op class 0  */
	0xFFC0, 0x00C0, ochk2,	"b",
	0xFF00, 0x0000, oimmed,	"or",
	0xFFC0, 0x02C0, ochk2,	"w",
	0xFF00, 0x0200, oimmed,	"and",
	0xFFC0, 0x04C0, ochk2,	"l",
	0xFF00, 0x0400, oimmed,	"sub",
	0xFFF8, 0x06C0, oreg, 	"%8trtm%8t%n", 
	0xFFC0, 0x06C0, ocallm, 0,
	0xFF00, 0x0600, oimmed,	"add",
	0xFFFF, 0x0AFC, ocas2,	"b",
	0xFFC0, 0x0AC0, ocas,	"b",
	0xFF00, 0x0A00, oimmed,	"eor",
	0xFFFF, 0x0CFC, ocas2,	"w",
	0xFFC0, 0x0CC0, ocas,	"w",
	0xFF00, 0x0C00, oimmed,	"cmp",
	0xFFFF, 0x0EFC, ocas2,	"l",
	0xFFC0, 0x0EC0, ocas,	"l",
	0xFFC0, 0x0E00, omoves,	"b",
	0xFFC0, 0x0E40, omoves,	"w",
	0xFFC0, 0x0E80, omoves,	"l",
	0xF138, 0x0108, omovep,	"movep",
	0xF100, 0x0100, biti,	0,
	0xF800, 0x0800, biti,	0,

	/* op class 4 */
	0xFFC0, 0x40C0, mvspec,	"sr,",
	0xFF00, 0x4000, soneop,	"negx",
	0xFFC0, 0x42C0, mvspec,	"cc,",
	0xFF00, 0x4200, soneop,	"clr",
	0xFFC0, 0x44C0, mvspec,	",cc",
	0xFF00, 0x4400, soneop,	"neg",
	0xFFC0, 0x46C0, mvspec,	",sr",
	0xFF00, 0x4600, soneop,	"not",
	0xFFF8, 0x4808, olink,	0,
	0xFFC0, 0x4800, oneop,	"nbcd",
	0xFFF8, 0x4840, oreg,	"%8tswap%8t%n",
	0xFFF8, 0x4848, oreg,	"%8tbkpt%8t#%r",
	0xFFC0, 0x4840, oneop,	"pea",
	0xFFF8, 0x4880, oreg,	"%8textw%8t%n",
	0xFFF8, 0x48C0, oreg,	"%8textl%8t%n",
	0xFFF8, 0x49C0, oreg,	"%8textbl%8t%n",
	0xFB80, 0x4880, omovem,	0,
	0xFFC0, 0x4AC0, oneop,	"tas",
	0xFF00, 0x4A00, soneop,	"tst",
	0xFFC0, 0x4C00, regpair,	"mul",
	0xFFC0, 0x4C40, regpair,	"div",
	0xFFF0, 0x4E40, otrap,	0,
	0xFFF8, 0x4E50, olink,	0,
	0xFFF8, 0x4E58, oreg,	"%8tunlk%8t%n",
	0xFFF8, 0x4E60, oreg, 	"%8tmove%8t%N",
	0xFFF8, 0x4e68, oreg,	"%8tmove%8tusp,%N",
	0xFFFF, 0x4E70, oprint,	"reset",
	0xFFFF, 0x4E71, oprint,	"nop",
	0xFFFF, 0x4E72, oconst,	"stop",
	0xFFFF, 0x4E73, oprint,	"rte",
	0xFFFF, 0x4E74, oconst,	"rtd",
	0xFFFF, 0x4E75, oprint,	"rts",
	0xFFFF, 0x4E76, oprint,	"trapv",
	0xFFFF, 0x4E77, oprint,	"rtr",
	0xFFFE, 0x4E7A, omovec, 0,
	0xFFC0, 0x4E80, oneop,	"jsr",
	0xFFC0, 0x4EC0, oneop,	"jmp",
	0xF1C0, 0x4180, ochk,	"w",
	0xF1C0, 0x4100, ochk,	"l",
	0xF1C0, 0x41C0, olea,	"lea",

	0xF0F8, 0x50C8, odbcc,	0,
	0xF0FF, 0x50FA, otrapcc,	0,
	0xF0FF, 0x50FB, otrapcc,	0,
	0xF0FF, 0x50FC, otrapcc,	0,
	0xF0C0, 0x50C0, oscc,	0,
	0xF100, 0x5000, oquick,	"addq",
	0xF100, 0x5100, oquick,	"subq",

	0xF000, 0x7000, omoveq,	0,

	0xF1C0, 0x80C0, odiv,	"divu",
	0xF1F0, 0x8100, extend,	"sbcd",
	0xF1F0, 0x8140, opack,	"pack",
	0xF1F0, 0x8180, opack,	"unpk",
	0xF1C0, 0x81C0, odiv,	"divs",
	0xF000, 0x8000, opmode,	"or",

	0xF1C0, 0x91C0, opmode,	"sub",
	0xF130, 0x9100, extend,	"subx",
	0xF000, 0x9000, opmode,	"sub",

	0xF1C0, 0xB1C0, opmode,	"cmp",
	0xF138, 0xB108, extend,	"cmpm",
	0xF100, 0xB000, opmode,	"cmp",
	0xF100, 0xB100, opmode,	"eor",

	0xF1C0, 0xC0C0, odiv,	"mulu",
	0xF1C0, 0xC1C0, odiv,	"muls",
	0xF1F8, 0xC188, extend,	"exg",
	0xF1F8, 0xC148, extend,	"exg",
	0xF1F8, 0xC140, extend,	"exg",
	0xF1F0, 0xC100, extend,	"abcd",
	0xF000, 0xC000, opmode,	"and",

	0xF1C0, 0xD1C0, opmode,	"add",
	0xF130, 0xD100, extend,	"addx",
	0xF000, 0xD000, opmode,	"add",

	0xF9C0, 0xE8C0, bfoneop, 0,
	0xF9C0, 0xE9C0, bftwoop, 0,
	0xF100, 0xE000, shift,	"r",
	0xF100, 0xE100, shift,	"l",

	0xFFFF, 0xF27A, ftrapcc,	0,
	0xFFFF, 0xF27B, ftrapcc,	0,
	0xFFFF, 0xF27C, ftrapcc,	0,
	0xFFF8, 0xF248, fdbcc,	0,
	0xFFC0, 0xF240, fscc,	0,
	0xFFC0, 0xF280, fbcc,	"w",
	0xFFC0, 0xF2C0, fbcc,	"l",
	0xFFC0, 0xF300, oneop,	"fsave",
	0xFFC0, 0xF340, oneop,	"frestore",
	0xFE00, 0xF200, fops,	0,
	0, 0, 0, 0
};


printins(idsp, ins)
REG INT ins;
{
	REG struct opdesc *p;

	type = DSYM; space=idsp; incp=2;
	IF ins==0 THEN printf("%8t.word 0"); return; FI
	FOR p = opdecode; p->mask; p++
	DO IF (ins&p->mask)==p->match
	   THEN	break;
	   FI
	OD
	IF p->opfun==NULL
	THEN	printf("%8t%X", ins);
	ELSE	(*p->opfun)(ins, p->farg);
	FI
	dotinc=incp;
}

sxtword (i)
register int i;
{
	return (i > 32767 ? i - 65536 : i);
}

instfetch (size)
{
	REG L_INT a;

	a = chkget(inkdot(incp), space);/* should be passed down from printins */
	incp += size;
	IF size==2
	THEN a = (a>>16)&0xFFFF;
	FI
	return a;
#ifdef notdef
	register int ans = 0;
	register int s;

	s = size;
	while (s--) {
		ans <<= 8;
		ans |= *(char *)inst_ptr++ & 0377;

/* TEST REMOVE  */
		dot++;
	}
	if (!codeonly) {
		if (size == 2)
			{
			printf ("%04x ", (unsigned short) ans);
			flushbuf();
			}
		else
			{
			printf ("%08lx ", ans);
			flushbuf();
			}
		clx += size * 2 + 1;
	}
	return (ans);
#endif
}

printea (mode, reg, size)
int mode, reg;
int size;
{
	int value;
	union kludge {
		double d;
		float f;
		long l[2]
	} kluj;

	switch (mode) {
	case 0: 
		printf("%n", reg);
		break;
	case 1: 
		printf("%N", reg);
		break;
	case 2: 
		printf("%N@", reg);
		break;
	case 3: 
		printf("%N@+", reg);
		break;
	case 4: 
		printf("%N@-", reg);
		break;
	case 5: 
		printf("%N@(", reg);
		printf("%r)", sxtword(instfetch(2)));
		break;
	case 6: 
		indexwd (reg);
		break;
	case 7: 
		switch (reg) {
		case 0: 
			value = instfetch (2);
			psymoff(value, type, ":w");
			break;
		case 1: 
			value = instfetch (4);
			psymoff(value, type, "");
			break;
		case 2: 
			printf("pc@(");
			psymoff(sxtword(instfetch(2)+2), type, ")");
			break;
		case 3: 
			indexwd (16);
			break;
		case 4: 
			if (size <= 2)
				printf("#%x", instfetch(2));
			else if (size == 4) {
				printf("#%r", instfetch(4));
			} else if (size == 6) {
				/* floating single precision */
				/* KLUDGE: size is really 4,
				 * but need to distinguish from long case 
				 */
				kluj.l[0] = instfetch(4);
				printf("#0f%g", kluj.f);
			} else if (size == 8) {
				/* floating double precision */
				kluj.l[0] = instfetch(4);
				kluj.l[1] = instfetch(4);
				printf("#0d%g", kluj.d);
			} else if (size == 12) {
				/* Floating extended and packed decimal */
				/* can't give real value, so do in hex */
				printf("#%x", instfetch(4));
				printf("%x", instfetch(4));
				printf("%x", instfetch(4));
			}
			break;
		default: 
			printf("???");
			break;
		}
		break;
	default: 
		printf("???");
	}
}

printEA (ea)
int ea;
{
	printea ((ea >> 3) & 07, ea & 07, 0);
}

union index_word {
	unsigned short word;
	struct {
		unsigned short isareg: 1;
		unsigned short reg: 3;
		unsigned short sz: 1;
		unsigned short scale: 2;
		unsigned short usefull: 1;
		unsigned short disp: 8;
	} brief;
	struct {
		unsigned short isareg: 1;
		unsigned short reg: 3;
		unsigned short sz: 1;
		unsigned short scale: 2;
		unsigned short usefull: 1;
		unsigned short bsupp: 1;
		unsigned short isupp: 1;
		unsigned short bdsize: 2;
		unsigned short zfill: 1;
		unsigned short ixsel: 3;
	} full;
};

indexwd(reg)
L_INT	reg;
{
	L_INT bdisp, odisp;
	union index_word ixwd;

	ixwd.word = instfetch (2);
	IF ixwd.brief.usefull == 0
	THEN
		bdisp = (char) ixwd.brief.disp;
		IF reg == 16
		THEN bdisp += 2;
		     printf("pc@");
		ELSE
			printf("%n", reg);
		FI
		printf("(");
		printf("%r,", bdisp);
		IF ixwd.brief.isareg
		THEN printf("%N", ixwd.brief.reg);
		ELSE printf("%n", ixwd.brief.reg);
		FI
		printf("%c*%c)", ixwd.brief.sz ? 'l' : 'w', 
			"1248"[ixwd.brief.scale]);
		return;
	FI

	/* now comes the really groady part */
	IF   reg == 16
	THEN printf("%spc@(", (ixwd.full.bsupp ? "z" : ""));
	ELIF ixwd.full.bsupp
	THEN printf("za@(");
	ELSE printf("%N", reg);
	FI
	switch (ixwd.full.bdsize) {
	case 0: 		/* this is really an error, but oh, well... */
	case 1: 
		printf("0");
		break;
	case 2: 
		bdisp = sxtword (instfetch (2));
		printf("%r", bdisp);
		break;
	case 3: 
		bdisp = instfetch (4);
		printf("%r", bdisp);
		break;
	}
	IF ixwd.full.ixsel & 04
	THEN
		printf(")[%c%d:%c*%c,",
			(ixwd.full.isareg ? 'a' : 'd'),
			ixwd.full.reg, (ixwd.full.sz ? 'l' : 'w'),
			"1248"[ixwd.full.scale]);
	ELSE
		IF ixwd.full.isupp
		THEN printf(")");
		ELSE printf(",%c%d:%c*%c)",
				(ixwd.full.isareg ? 'a' : 'd'),
				ixwd.full.reg, (ixwd.full.sz ? 'l' : 'w'),
				"1248"[ixwd.full.scale]);
		FI
		IF ixwd.full.ixsel != 0
		THEN printf("[");
		FI
	}
	IF ixwd.full.ixsel != 0
	THEN
		switch (ixwd.full.ixsel & ~04) {
		case 0: 	/* this is really an error, but oh, well... */
		case 1: 
			printf("0]");
			break;
		case 2: 
			odisp = sxtword (instfetch (2));
			printf("%r]", odisp+2);
			break;
		case 3: 
			odisp = instfetch (4);
			printf("%r]", odisp+2);
			break;
		}
	}
}

mapsize (inst)
int inst;
{
	inst >>= 6;
	inst &= 03;
	return ((inst == 0) ? 1 : (inst == 1) ? 2 : (inst == 2) ? 4 : -1);
}

char
suffix (size)
register int size;
{
	return ((size == 1) ? 'b' : (size == 2) ? 'w' :
		(size == 4) ? 'l' : '?');
}

omove(inst, s)
int inst;
char *s;
{
	int size;

	printf("%8tmov%c%8t", *s);
	size = ((*s == 'b') ? 1 : (*s == 'w') ? 2 : 4);
	printea ((inst >> 3) & 07, inst & 07, size);
	printf(",");
	printea ((inst >> 6) & 07, (inst >> 9) & 07, size);
}

obranch (inst, dummy)
int inst;
{
	int disp = inst & 0377;
	int cond = (inst >> 8) & 017;
	char len;

	len = 's';
	if (disp == 0) {
		len = ' ';
		disp = sxtword (instfetch (2));
	} else if (disp == 255) {
		len = 'l';
		disp = instfetch (4);
	} else if (disp > 127) {
		disp |= ~0377;
	}
	printf("%8tb%s%c%8t", brname[cond], len);
	psymoff(dot + 2 + disp, type, "");
	return;
}

odbcc (inst, dummy)
int inst;
{
	int disp;
	register int cond = (inst >> 8) & 017;

	disp = sxtword (instfetch (2)) - 2;
	printf("%8tdb%s%8td%d,",
 		(cond == 1 ? "ra" : condname[cond]),
		inst & 07);
	psymoff(dot+disp+2, type, "");
}

oscc (inst, dummy)
int inst;
{
	register int cond = (inst >> 8) & 017;

	printf("%8ts%s%8t", condname[cond]);
	printea ((inst >> 3) & 07, inst & 07, 1);
}

biti (inst, dummy)
int inst;
{
	printf("%8tb%s%8t", opname[((inst >> 6) & 03)]);
	if (inst & 0x0100)
		printf("d%d,", (inst >> 9) & 07);
	else
		printf("#%d,", instfetch (2));
	printEA (inst);
}

opmode (inst, opcode)
int inst;
{
	register int mode = ((inst >> 6) & 07);
	register int reg = ((inst >> 9) & 07);
	int size;

	size = (mode == 0 || mode == 4) ? 1 :
		(mode == 1 || mode == 3 || mode == 5) ? 2 : 4;
	printf("%8t%s%c%8t", opcode, suffix(size));
	if (mode >= 4 && mode <= 6) {
		printf("d%d,", reg);
		printea ((inst >> 3) & 07, inst & 07, size);
	} else {
		printea ((inst >> 3) & 07, inst & 07, size);
		printf(",%c%d", (mode <= 2) ? 'd' : 'a', reg);
	}
}

shift (inst, ds)
int inst;
char *ds;
{
	int rx, ry;
	char *opcode;
	if ((inst & 0xC0) == 0xC0) {
		opcode = shname[((inst >> 9) & 03)];
		printf("%8t%s%s%8t", opcode, ds);
		printEA (inst);
	} else {
		opcode = shname[((inst >> 3) & 03)];
		printf("%8t%s%s%c%8t", opcode, ds, suffix (mapsize (inst)));
		rx = ((inst >> 9) & 07);
		ry = (inst & 07);
		if (inst & 0x0020)
			printf("d%d,d%d", rx, ry);
		else
			printf("#%d,d%d", (rx ? rx : 8), ry);
	}
}

oimmed (inst, opcode)
int inst;
register char *opcode;
{
	register int size = mapsize (inst);
	register int mode = (inst >> 3) & 07;
	register int reg = inst & 07;
	int const;

	if (size > 0) {
		const = instfetch (size == 4 ? 4 : 2);
		printf("%8t%s%c%8t#", opcode, suffix(size));
		if (size != 4)
			printf("%R", const);
		else
			psymoff(const, type, "");
		printf(",");
		if (mode == 7 && reg == 4)/* ccr or sr is destination */
			printf(size == 1 ? "cc" : "sr");
		else
			printea (mode, reg, size);
	} else
		printf(badop);
}

oreg (inst, opcode)
int inst;
register char *opcode;
{
	printf(opcode, inst & 07);
}

extend (inst, opcode)
int inst;
char *opcode;
{
	register int size = mapsize (inst);
	int ry = (inst & 07), rx = ((inst >> 9) & 07);
	char c;

	c = ((inst & 0x1000) ? suffix (size) : ' ');
	printf("%8t%s%c%8t", opcode, c);
	if (*opcode == 'e') {
		if (inst & 0x0080)
			printf("%n, %N", rx, ry);
		else if (inst & 0x0008)
			printf("%N,%N", rx, ry);
		else
			printf("%n,%n", rx, ry);
	} else if ((inst & 0xF000) == 0xB000)
		printf("%N@+, %N@+", ry, rx);
	else if (inst & 0x8)
		printf("%N@-,%N@-", ry, rx);
	else
		printf("%n, %n", ry, rx);
}

olink (inst, dummy)
int inst;
{
	int disp;

	if (inst & 0x8)
		disp = instfetch (4);
	else
		disp = sxtword (instfetch (2));
	printf("%8tlink%8t%N,#%r", inst & 07, disp);
}

otrap (inst, dummy)
int inst;
{
	printf("%8ttrap%8t#%r", inst & 017);
}

mvspec (inst, opcode)
int inst;
char *opcode;
{
	printf("%8tmovw%8t");
	IF *opcode == ','
	THEN printEA(inst); prints(opcode);
	ELSE prints(opcode); printEA(inst);
	FI
}

oneop (inst, opcode)
int inst;
char *opcode;
{
	printf("%8t%s%8t", opcode);
	printEA (inst);
}

pregmask (mask)
register int mask;
{
	register int i;
	register int flag = 0;

	printf("#<");
	for (i = 0; i < 16; i++) {
		if (mask & 1) {
			if (flag)
				printf(",");
			else
				flag++;
			printf("%c%d", (i < 8) ? 'd' : 'a', i & 07);
		}
		mask >>= 1;
	}
	printf(">");
}

omovem (inst, dummy)
int inst;
{
	register int i, list = 0, mask = 0100000;
	register int reglist = instfetch (2);

	IF (inst&070)==040		/* predecrement */
	THEN
		FOR i=15; i>0; i-=2
		DO
			list |= ((mask & reglist) >> i);
			mask >>= 1;
		OD
		FOR i=1; i<16; i+=2
		DO
			list |= ((mask & reglist) << i);
			mask >>= 1;
		OD
		reglist = list;
	FI
	printf("%8tmovem%c%8t", (inst & 0100) ? 'l' : 'w');
	IF inst&02000
	THEN
		printEA (inst);
		printf(",");
		pregmask(reglist);
	ELSE
		pregmask (reglist);
		printf(",");
		printEA (inst);
	FI
}

ochk (inst, s)
int inst;
char *s;
{
	printf("%8tchk%s%8t", s);
	printea ((inst >> 3) & 07, inst & 07, (*s == 'w') ? 2 : 4);
	printf(",%n", (inst >> 9) & 07);
}

odiv (inst, opcode)
int inst;
char *opcode;
{
	printf("%8t%s%8t", opcode);
	printEA (inst);
	printf(",%n", (inst >> 9) & 07);
}

olea (inst, dummy)
int inst;
{
	printf("%8tlea%8t");
	printEA (inst);
	printf(",%N", (inst >> 9) & 07);
}

soneop (inst, opcode)
int inst;
register char *opcode;
{
	register int size = mapsize(inst);

	IF size>0
	THEN
		printf("%8t%s%c%8t", opcode, suffix(size));
		printEA (inst);
	ELSE
		printf(badop);
	FI
}

oquick (inst, opcode)
int inst;
register char *opcode;
{
	register int size = mapsize(inst);
	register int data = ((inst >> 9) & 07);

	IF data == 0 THEN data = 8; FI
	IF size > 0
	THEN printf("%8t%s%c%8t#%r,", opcode, suffix(size), data);
	     printEA (inst);
	ELSE printf(badop);
	FI
}

omoveq (inst, dummy)
int inst;
{
	register int data = (unsigned char) (inst & 0377);

	printf("%8tmoveq%8t#%r,%n", data, (inst >> 9) & 07);
}

oprint (inst, opcode)
int inst;
register char *opcode;
{
	printf("%8t%s", opcode);
}

oconst (inst, opcode)
int inst;
register char *opcode;
{
	printf("%8t%s%8t#%r", opcode, (unsigned int)instfetch(2));
}

omovep (inst, opcode)
int inst;
char *opcode;
{
	int areg = inst & 07;
	int dreg = (inst >> 9) & 07;
	int disp = sxtword (instfetch (2));

	printf("%8t%s%c%8t", opcode, inst & 0100 ? 'l' : 'w');
	IF inst & 0200
	THEN printf("%n, %N@(%r)", dreg, areg, disp);
	ELSE printf("%N@(%r), %n", areg, disp, dreg);
	FI
}

char *
getctlreg (op2)
int op2;
{
	switch (op2 & 0x0FFF) {
	case 0x0000: 
		return ("sfc");
	case 0x0001: 
		return ("dfc");
	case 0x0002: 
		return ("cac");
	case 0x0800: 
		return ("usp");
	case 0x0801: 
		return ("vbr");
	case 0x0802: 
		return ("caa");
	case 0x0803: 
		return ("msp");
	case 0x0804: 
		return ("isp");
	default: 
		return ("???");
	}
}

omovec (inst, dummy)
int inst;
char *dummy;
{
	char regtype, *ctlreg;
	int regnumb;
	int info = instfetch (2);

	regtype = (info & 0x8000) ? 'a' : 'd';
	regnumb = (info >> 12) & 07;
	ctlreg = getctlreg (info);
	printf("%8tmovec%8t");
	IF inst & 01
	THEN printf("%c%d,%s", regtype, regnumb, ctlreg);
	ELSE printf("%s,%c%d", ctlreg, regtype, regnumb);
	FI
}

omoves (inst, s)
int inst;
char *s;
{
	char regtype;
	int regnumb, size;
	int info = instfetch (2);

	regtype = (info & 0x8000) ? 'a' : 'd';
	regnumb = (info >> 12) & 07;
	size = ((*s == 'b') ? 1 : (*s == 'w') ? 2 : 4);
	printf("%8tmoves%c%8t", *s);
	IF info & 0x8000
	THEN printf("%c%d, ", regtype, regnumb);
	     printea ((inst >> 3) & 07, inst & 07, size);
	ELSE printea((inst >> 3) & 07, inst & 07, size);
	     printf(",%c%d", regtype, regnumb);
	FI
}

bfoneop (inst, dummy)
int inst;
{
	unsigned short extwd = instfetch (2);

	printf("%8tbf%s%8t", opname[(inst>>9) & 03]);
	printEA (inst);
	if (extwd & 0x0800)
		printf("{%n:", (extwd >> 6)&08);
	else
		printf("{%r:", (extwd>>6) & 0x1f);
	if (extwd & 0x0020)
		printf("%n}", extwd & 07);
	else
		printf("%r}", extwd & 0x1f);
}

bftwoop (inst, dummy)
int inst;
{
	unsigned short extwd = instfetch (2);
	short bfins = 0;
	char *opcode;

	switch ((inst >> 9) & 03) {
	case 0: 
		opcode = "extu";
		break;
	case 1: 
		opcode = "exts";
		break;
	case 2: 
		opcode = "ffo";
		break;
	case 3: 
		opcode = "ins";
		bfins++;
		break;
	}
	printf("%8tbf%s%8t", opcode);
	IF bfins THEN printf("%n, ", (extwd >> 12) & 07); FI
	printEA (inst);
	IF extwd & 0x0800
	THEN printf("{%n:", (extwd>>6)&07);
	ELSE printf("{$r:", (extwd>>6)&0x1f);
	FI
	IF extwd & 0x0020
	THEN printf("%n}", extwd&07);
	ELSE printf("%r}", extwd&0x1f);
	FI
	IF bfins == 0 THEN printf(",%n", (extwd>>12)&07); FI
}

ocallm (inst, dummy)
int inst;
{
	printf("%8tcallm%8t#%r, ", instfetch(2));
	printEA (inst);
}

ocas (inst, s)
int inst;
char *s;
{
	unsigned short extwd = instfetch (2);

	printf("%8tcas%s%8td%d,d%d,", s, extwd & 07, (extwd >> 6) & 07);
	printEA (inst);
}

ocas2 (inst, s)
int inst;
char *s;
{
	unsigned short extwd1, extwd2;
	int r1, r2;

	extwd1 = instfetch (2);
	extwd2 = instfetch (2);

	r1 = (extwd1 & 0x8000) ? 'a' : 'd';
	r2 = (extwd2 & 0x8000) ? 'a' : 'd';
	printf("%8tcas2%s%8td%d:d%d,d%d:d%d,%c%d@:%c%d@",
		s, extwd1 & 07, extwd2 & 07,
		(extwd1 >> 6) & 07, (extwd2 >> 6) & 07,
		r1, (extwd1 >> 12) & 07, r2, (extwd2 >> 12) & 07);
}

ochk2 (inst, s)
int inst;
char *s;
{
	unsigned short extwd = instfetch (2);

	printf("%8t%s%s%8t", (extwd & 0x0800) ? "chk2" : "cmp2", s);
	printea ((inst >> 3) & 07, inst & 07,
		(*s == 'b') ? 1 : (*s == 'w') ? 2 : 4);
	printf(",%c%d", (extwd & 0x8000) ? 'a' : 'd', (extwd >> 12) & 07);
}

opack (inst, opcode)
int inst;
char *opcode;
{
	int rx = (inst & 07), ry = ((inst >> 9) & 07);

	printf("%8t%s%8t", opcode);
	if (inst & 0x8)
		printf("%N@-,%N@-", rx, ry);
	else
		printf("d%d,d%d", rx, ry);
	printf(",#%r", instfetch(2));
}

otrapcc (inst, dummy)
int inst;
{
	register int cond = (inst >> 8) & 017;
	char s;
	int val;

	switch (inst & 07) {
	case 2: 
		val = instfetch (2);
		s = 'w';
		break;
	case 3: 
		val = instfetch (4);
		s = 'l';
		break;
	case 4: 
		s = ' ';
		break;
	}
	printf("%8ttrap%s%c", condname[cond], s);
	printf("%8t#%r", val);
}

regpair (inst, opcode)
int inst;
char *opcode;
{
	register short extwd = instfetch (2);
	int reg1, reg2;
	char us, *sz, havepr;

	us = (extwd & 0x0800) ? 's' : 'u';
	reg1 = (extwd >> 12) & 07;
	reg2 = extwd & 07;
	sz = "l";
	if (extwd & 0x0400)
		havepr = 1;
	else if (reg1 != reg2) {
		/* div?ll case */
		sz = "ll";
		havepr = 1;
	} else
		havepr = 0;

	printf("%8t%s%c%s%8t", opcode, us, sz);
	printea ((inst >> 3) & 07, inst & 07, 4);
	if (havepr) {
		printf(", %n:%n", reg2, reg1);
	} else
		printf(",%n", reg1);
}

/* 68881 floating point processor routines */

char fformat[] = "lsxpwdbp";
/* for formats:	l  s  x   p   w  d  b  p?	*/
int fsize[] = { 4, 6, 12, 12, 2, 8, 1, 0};
/* fsize[s] == 6 is a kludge to distinguish 'l' and 's' immediates */

char *fopname[] = {
	/* zero entries are redundant encodings and will be flagged */
	"mov", "int", "sinh", "intrz",
	"sqrt",	0, "lognp1", 0,
	"etoxm1", "tanh", "atan", 0, 
	"asin", "atanh", "sin", "tan",
	"etox", "twotox", "tentox", 0,
	"logn", "log10", "log2", 0,
	"abs", "cosh", "neg", 0,
	"acos", "cos", "getexp", "getman",
	"div", "mod", "add", "mul",
	"sgldiv", "rem", "scale", "sglmul",
	"sub", 0, 0, 0,
	0, 0, 0, 0,
	"", "", "", "", 			/* sincos is special */
	"", "", "", "", 			/* sincos is special */
	"cmp", 0, "tst", 0,
	0, 0, 0, 0,
	/* 0x40 to 0x7f are reserved and checked elsewhere */
};

fops (inst, dummy)
int inst;
{
	int cmd = instfetch (2);

	switch ((cmd >> 13) & 07) {
	case 0:	fop_reg_reg (inst, cmd); break;
	case 1: /* reserved encoding */
		printf("%8tfreserved???");
		break;
	case 2:	fop_ea_reg (inst, cmd); break;
	case 3: fmov_ea (inst, cmd); break;
	case 4: fmovec (inst, cmd, 0); break;
	case 5: fmovec (inst, cmd, 1); break;
	case 6: fmovem (inst, cmd, 0); break;
	case 7: fmovem (inst, cmd, 1); break;
	}
}

fop_reg_reg (inst, cmd)
int inst, cmd;
{
	int op = cmd & 0x7f;

	if (op >= 0x40 || fopname[op] == 0) {
		printf("%8tfreserved???");
		return;
	}
	if (fopname[op][0] == '\0') {
		/* handle fsincosx op */
		printf("%8tfsincosx%8t%d, f%d: f%d", (cmd >> 10) & 07,
			cmd & 07, (cmd >> 7) & 07);
		return;
	}
	if (op == 0x3A) {
		/* handle ftstx op */
		printf("%8tftstx%8tf%d", (cmd >> 10) & 07);
		return;
	}
	/* finally all others */
	printf("%8tf%sx%8tf%d,f%d",
		fopname[op], (cmd >> 10) & 07, (cmd >> 7) & 07);
}

fop_ea_reg (inst, cmd)
int inst, cmd;
{
	int op = cmd & 0x7f;
	int fmt = (cmd >> 10) & 07;
	short sincos = 0;
	char szch = fformat[fmt];

	if (fmt == 7 && (inst & 0xCF) == 0) {
		/* handle fmovecr op */
		printf("%8tfmovecr%8t#%d,f%d", op, (cmd >> 7) & 07);
		return;
	}

	if (op >= 0x40 || fopname[op] == 0) {
		printf("%8tfreserved???");
		return;
	}
	if (fopname[op][0] == '\0') {
		/* handle fsincos op */
		printf("%8tfsincos%c%8t", szch);
		sincos++;
	} else
		printf("%8tf%s%c%8t", fopname[op], szch);
	printea ((inst >> 3) & 07, inst & 07, fsize[fmt]);
	if (op != 0x3A) {
		/* all except ftst op have destination */
		printf(",f%d", (cmd >> 7) &07);
		if (sincos) {
			/* fsincos has two destination */
			printf(": f%d", cmd & 07);
		}
	}
}

fmov_ea (inst, cmd)
int inst, cmd;
{
	int kfactor = cmd & 0x7f;
	int fmt = (cmd >> 10) & 07;

	printf("%8tfmov%c%8tf%d,", fformat[fmt], (cmd >> 7) & 07);
	printea ((inst >> 3) & 07, inst & 07, fsize[fmt]);

	if (fmt == 3) {
		/* need to sign extend the number */
		if (kfactor > 63)
			kfactor |= ~0x3f;
		printf("{%d}", kfactor);
	} else if (fmt == 7) {
		printf("{d%d}", (kfactor >> 4) & 07);
	}
}

fmovec (inst, cmd, dir)
int inst, cmd, dir;
{
	register int reglist = (cmd >> 10) & 07;
	char *creg;

	switch (reglist) {
	case 0:
		printf("%8tfreserved???"); return;
	case 1:
		creg = "fpiar"; break;
	case 2:
		creg = "fpsr"; break;
	case 4:
		creg = "fpcr"; break;
	default:
		printf("%8tfmoveml%8t");
		if (dir == 0) {
			printEA (inst);
			printf(",");
			fcregmask (reglist);
		} else {
			fcregmask (reglist);
			printf(",");
			printEA (inst);
		}
		return;
	}

	printf("%8tfmovel%8t");
	if (dir == 0) {
		printea ((inst >> 3) & 07, inst & 07, 4);
		printf(",%s", creg);
	} else {
		printf("%s,", creg);
		printEA (inst);
	}
}

fcregmask (mask)
register int mask;
{
	short flag = 0;

	printf("#<");
	if (mask & 04) {
		printf("fprc");
		flag++;
	}
	if (mask & 02) {
		if (flag)
			printf(",");
		printf("fpsr");
		flag++;
	}
	if (mask & 01) {
		if (flag)
			printf(",");
		printf("fpiar");
	}
	printf(">");
}

fmovem (inst, cmd, dir)
int inst, cmd, dir;
{
	register int reglist = cmd & 0xff;

	if (cmd & 0x0800) {
		/* register list is in a data register */
		reglist = (reglist >> 4) & 07;
	} else if ((inst & 070) == 040) {/* predecrement */
		register int i, list, mask;

		list = 0; mask = 0x80;
		for (i = 7; i > 0; i -= 2) {
			list |= ((mask & reglist) >> i);
			mask >>= 1;
		}
		for (i = 1; i < 8; i += 2) {
			list |= ((mask & reglist) << i);
			mask >>= 1;
		}
		reglist = list;
	}
	printf("%8tfmovemx%8t");
	if (dir == 0) {
		printEA (inst);
		printf(",");
		if (cmd & 0x0800)
			printf("%n", reglist);
		else
			fregmask (reglist);
	} else {
		if (cmd & 0x0800)
			printf("%n", reglist);
		else
			fregmask (reglist);
		printf(",");
		printEA (inst);
	}
}

fregmask (mask)
register int mask;
{
	register int i;
	register int flag = 0;

	printf("#<");
	for (i = 0; i < 8; i++) {
		if (mask & 0x80) {
			if (flag)
				printf(",");
			else
				flag++;
			printf("f%d", i);
		}
		mask <<= 1;
	}
	printf(">");
}

char *fcondname[32] = {
	"f",    "eq",  "ogt", "oge", "olt", "ole", "ogl",  "or",
	"un",   "ueq", "ugt", "uge", "ult", "ule", "neq",  "t",
	"sf",   "seq", "gt",  "ge",  "lt",  "le",  "gl",   "gle",
	"ngle", "ngl", "nle", "nlt", "nge", "ngt", "sneq", "st",
};

fbcc (inst, DAS_ling)
int inst;
char *DAS_ling;
{
	int disp;
	int cond = inst & 0x3F;

	if (*DAS_ling == 'w') {
		disp = sxtword (instfetch (2)) - 2;
	} else if (*DAS_ling == 'l') {
		disp = instfetch (4) - 2;
	}

	if (cond == 0 && *DAS_ling == 'w' && disp == -2)
		printf("%8tfnop");
	else
		printf("%8tb%s%c%8t%r",
			fcondname[cond], *DAS_ling, dot + disp);
}

fdbcc (inst, dummy)
int inst;
{
	int disp;
	register int cond = instfetch (2);

	disp = sxtword (instfetch (2)) - 2;
	printf("%8tdb%s%8t%n, %r",
		fcondname[cond], inst & 07, dot + disp);
}

fscc (inst, dummy)
int inst;
{
	register int cond = instfetch (2);

	printf("%8tfs%s%8t", fcondname[cond]);
	printea ((inst >> 3) & 07, inst & 07, 1);
}

ftrapcc (inst, dummy)
int inst;
{
	register int cond = instfetch (2);
	char s;
	int val;

	switch (inst & 07) {
	case 2: 
		val = instfetch (2);
		s = 'w';
		break;
	case 3: 
		val = instfetch (4);
		s = 'l';
		break;
	case 4: 
		s = ' ';
		break;
	}
	printf("%8tftrap%s%c", fcondname[cond], s);
	if (s == 'w')
		printf("%8t#%r", val);
	else if (s == 'l')
		printf("%8t#%r", val);
}
