/*
 * libc/gen/decvax_d.c
 */

/*
 * decvax_d(double *ddp, double *idp)
 * Convert double from IEEE format to DECVAX format.
 * idp points to an IEEE format double to convert.
 * ddp points to a destination for the converted DECVAX value;
 * ddp may be identical to idp for in-place conversion.
 * Returns 0 on success, -1 on underflow, 1 on overflow.
 */

int
decvax_d(ddp, idp) double *ddp, *idp;
{
	unsigned char a[8];
	register unsigned char *cp;
	register int exp, a0123456;

	/* Extract sign, biased exponent. */
	cp = (char *)idp;
	exp = ((cp[7] & 0x7F) << 4) | ((cp[6] & 0xF0) >> 4);
	if (exp == 0) {
		a0123456 = (a[6] & 0x0F) | a[5] | a[4] | a[3] | a[2] | a[1] | a[0];
		memset((char *)ddp, 0, 8);		/* set to 0.0 */
		return (a0123456 == 0) ? 0 : -1;	/* return -1 for denormals */
	}

	/* Adjust exponent bias. */
	exp -= 1023 - 129;
	if (exp <= 0) {
		memset((char *)ddp, 0, 8);	/* set to 0.0 */
		return -1;			/* underflow */
	} else if (exp > 255) {
		/* Overflow, also used for NaNs. */
		cp = (char *)ddp;
		memset(cp, 0xFF, 7);		/* set to max */
		cp[7] |= 0x7F;			/* preserving sign bit */
		return 1;			/* overflow */
	}

	/* Repack. */
	a[7] = (cp[7] & 0x80) | ((exp & 0xFE) >> 1);
	a[6] = ((exp & 1) << 7) | ((cp[6] & 0x0F) << 3) | ((cp[5] & 0xE0) >> 5);
	a[5] = ((cp[5] & 0x1F) << 3) | ((cp[4] & 0xE0) >> 5);
	a[4] = ((cp[4] & 0x1F) << 3) | ((cp[3] & 0xE0) >> 5);
	a[3] = ((cp[3] & 0x1F) << 3) | ((cp[2] & 0xE0) >> 5);
	a[2] = ((cp[2] & 0x1F) << 3) | ((cp[1] & 0xE0) >> 5);
	a[1] = ((cp[1] & 0x1F) << 3) | ((cp[0] & 0xE0) >> 5);
	a[0] = ((cp[0] & 0x1F) << 3);
	memcpy((char *)ddp, a, 8);
	return 0;
}

/* end of libc/gen/decvax_d.c */
