/*
 * Fast driver for DASDs with RPS.
 */
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/370.h"
#include "../h/io.h"
#include "../h/tty.h"
#include "../h/ioconf.h"
 
#define NTYPS   5               /* number of types of DASDs             */
#define NCP     16              /* number of simultaneous reads and     */
			        /* writes to one disk at a time         */
			        /* to get up to 65,536 bytes            */

#define MAXWRITE 65536          /* maximum write size                   */
#define CPSIZE   5              /* size of channel program for one block*/
#define FORMSIZE (2*MAXWRITE/BSIZE)+3
                                /* size of channel program for one track*/
                                /* worst case, MAXWRITE on one track    */
#define CCWCOUNT (MAXWRITE*5)/BSIZE
                                /* worst case number of ccws to do      */
                                /* MAXWRITE (one block/track)           */
 
/*
 * Channel commands
 */
#define SEEK     0x07
#define SEEKHEAD 0x1B
#define SETSCTR  0x23
#define SRCHIDE  0x31
#define TIC      0x08
#define RDDATA   0x06
#define WRTDATA  0x05
#define WRTCKD   0x1D
#define NOOP     0x03
 
/*
 * Sense bits
 */
#define stest(byte,bit) (sense[byte]&(1<<(7-bit)))
#define CMDREJ  stest(0,0)
#define INTREQ	stest(0,1)
#define BUSOUT	stest(0,2)
#define EQCHK	stest(0,3)
#define DATACHK	stest(0,4)
#define OVERRUN	stest(0,5)
#define PERMERR	stest(1,0)
#define ENVDATA	stest(2,3)
#define ENDOCYL stest(1,2)
#define NOREC   stest(1,4)
 
#define DSKPRI (PZERO+1)        /* open priority                        */
#define TRKSKP 1                /* tracks skipped on cyl 0              */
#define MAXCYL 1024             /* largest possible cyl #               */
#define MAXTRK MAXWRITE/BSIZE   /* largest number of tracks in 64k      */
#define MAXBLK MAXWRITE/BSIZE   /* largest number of blocks/track at    */
                                /* 64k This is ugly                     */

/*
 * DASD type descriptor
 */
struct fdskdesc {
        int     fd_blktrk;       /* blocks per track */
        int     fd_trkcyl;       /* tracks per cylinder */
        int     fd_const;        /* RPS */
        int     fd_factor;       /* RPS */
        int     fd_divsor;       /* RPS */
};
 
/*
 * DASD descriptor - must be double word aligned
 */
struct fdasd {
        ccw_t   fd_ccws[CCWCOUNT]; /* channel programs                  */
        ccw_t   *fd_tccw;          /* pointer to ccw of interest        */
	struct frecloc {           /* record location for channel programs */
	        short   fd_bb;     /* bin number (always 0)             */
		short   fd_cc;     /* cylinder number                   */
	        short   fd_hh;     /* head number                       */
		short   fd_r0;     /* record number                     */
	        char    fd_sector; /* RPS sector                        */
	}       fd_recloc[NCP];
	struct idaws {             /* Indirect addressing areas         */
		int fd_idaw[BSIZE/2048+1];
	}       fd_idaws[NCP];
	struct  cnt {              /* for count of formatted writes     */
                short   fd_count[4];
	}       fd_cnts[MAXBLK];
			           /* (points to block to be read by channel program fd_cpst) */
	struct  idaws   *fd_tidaw; /* pointer to current idaw list      */
	struct  cnt     *fd_tcnt;  /* pointer to current count list     */
	struct  frecloc *fd_trec;  /* pointer to current recloc         */
        struct  fdskdesc *fd_desc; /* pointer to type of DASD           */
        int     fd_devaddr;        /* device address                    */
	int     fd_error;          /* error indicator */
	int     fd_errct;          /* error count */
        unsigned fd_maxadd;        /* first bad read/write address      */
        char    fd_init;           /* initialization flag (0 = not open,       */
                                   /* 1 = open or close in progress, 2 = open) */
        char    fd_rw;             /* read/write flag (0 = read only, 1 = r/w) */
        char    fd_done;           /* synch i/o flag                    */
        char    fd_form;           /* formatting write flag             */
} fdasds[NDASD];
 
/*
 * DASD type table
 * Keep in descending order of # tracks
 */
struct fdskdesc fdskdesc[NTYPS] = {
        14858/(BSIZE+198),      768,    118,    BSIZE+198,      84,     /*2305*/
        14568/(BSIZE+432),      384,    234,    BSIZE+430,      168,    /*2305*/
        19254/(BSIZE+185),      30,     389,    BSIZE+185,      156,    /*3350*/
        13165/(BSIZE+135),      19,     237,    BSIZE+135,      105,    /*3330*/
        8535/(BSIZE+167),       12,     353,    BSIZE+167,      140,    /*3340*/
};
 
/*
 * Open a fast DASD device.
 * Determine type, check for writeability, find size, and initialize ccws.
 */
fdopen(dev, flag)
{
        register struct fdasd *fd;
	struct fdskdesc *dd, *fdsktype();
        int fdsktioi();
 
	if(minor(dev) >= NDASD) {
		u.u_error = ENXIO;
		return;
	}
        fd = &fdasds[minor(dev)];
        if(fd->fd_init) {
		u.u_error = EBUSY;
		return;
	}
        fd->fd_init = 1;
        fd->fd_form = 0;
        fd->fd_done = 0;
        fd->fd_devaddr = cdevsw[major(dev)].d_addrs[minor(dev)];
	if (fd->fd_devaddr == 0) {
		u.u_error = ENXIO;
                fd->fd_init = 0;
		return;
	}
        tio(fd->fd_devaddr, fdsktioi, (int)fd);
        while (fd->fd_done == 0)
		sleep((caddr_t)&fd->fd_done, DSKPRI);
        if (fd->fd_done != 1) {
                u.u_error = ENXIO;
                fd->fd_init = 0;
                return;
        }

        if ((dd = fdsktype(fd)) <= 0) {
                u.u_error = ENXIO;
                fd->fd_init = 0;
                return;
        }
        fd->fd_desc = dd;
        if (vm)
		fd->fd_rw = flag && d_rwdasd(fd->fd_devaddr);
        else
		fd->fd_rw = flag;
        if (flag && !fd->fd_rw) {
                u.u_error = EROFS;
                fd->fd_init = 0;
                return;
        }
        fd->fd_maxadd = (fdskcyls(fd) * dd->fd_trkcyl - TRKSKP) * dd->fd_blktrk * BSIZE;
	fdskel(fd);
        fd->fd_init = 2;
}
 
/*
 * Close DASD.
 */
fdclose(dev)
{
	fdasds[minor(dev)].fd_init = 0;
}
 
/*
 * Determine DASD type.
 * This is done by seeking to the highest track possible
 * for a type and seeing if it works.
 */
struct fdskdesc *
fdsktype(fd)
register struct fdasd *fd;
{
        int i, fdskiint();
 
        fd->fd_recloc[0].fd_bb = 0;
        fd->fd_recloc[0].fd_cc = 0;
        fd->fd_ccws[0].cc_dblw = 0;
        fd->fd_ccws[0].cc_cmd = SEEK;
        fd->fd_ccws[0].cc_addr = (int)&fd->fd_recloc[0].fd_bb;
        fd->fd_ccws[0].cc_count = 6;
        for (i=0; i < NTYPS; i++) {
                fd->fd_recloc[0].fd_hh = fdskdesc[i].fd_trkcyl - 1;
                fd->fd_done = 0;
                sio(fd->fd_devaddr, &fd->fd_ccws[0], fdskiint, (int)fd);
                while (fd->fd_done == 0)
			sleep((caddr_t)&fd->fd_done, DSKPRI);
                switch (fd->fd_done) {
                case 1:
                        return(&fdskdesc[i]);
                case 2:
                        break;
                case 3:
                        return(0);
                }
	}
        return(0);
}
 
/*
 * Determine the number of cylinders on a DASD.
 * This is done using a binary search of seeks.
 */
fdskcyls(fd)
register struct fdasd *fd;
{
        int fdskiint();
        register maxgood, minbad;

        maxgood = 0;
        minbad = MAXCYL;
        fd->fd_ccws[0].cc_dblw = 0;
        fd->fd_ccws[0].cc_cmd = SEEK;
        fd->fd_ccws[0].cc_addr = (int) &fd->fd_recloc[0].fd_bb;
        fd->fd_ccws[0].cc_count = 6;
        fd->fd_recloc[0].fd_bb = 0;
        fd->fd_recloc[0].fd_hh = 0;
        while (maxgood < minbad - 1) {
                fd->fd_recloc[0].fd_cc = (maxgood+minbad) / 2;
                fd->fd_done = 0;
                sio(fd->fd_devaddr, &fd->fd_ccws[0], fdskiint, (int)fd);
                while (fd->fd_done == 0)
			sleep((caddr_t)&fd->fd_done, DSKPRI);
                if (fd->fd_done == 1)
			maxgood = fd->fd_recloc[0].fd_cc;
                else
			minbad = fd->fd_recloc[0].fd_cc;
        }
        return(maxgood+1);
}
 
/*
 * DASD initialization interrupt.
 */
fdskiint(fd, csw, sense)
register struct fdasd *fd;
csw_t *csw;
char *sense;
{
        if (csw->cs_uc) {
		if(CMDREJ)
                        fd->fd_done = 2;
		else
                        fd->fd_done = 3;
        } else
                if (csw->cs_de)
                        fd->fd_done = 1;
        if (csw->cs_cc == 3)
                fd->fd_done = 3;
        if (fd->fd_done)
                wakeup((caddr_t)&fd->fd_done);
}
 
/*
 * Check results of tio.
 * ARGSUSED
 */
fdsktioi(fd, csw, sense)
register struct fdasd *fd;
csw_t *csw;
char *sense;
{
        if (csw->cs_cc == 3)
                fd->fd_done = 3;
        else if (csw->cs_uc)
                fd->fd_done = 2;
        else
                fd->fd_done = 1;
        wakeup((caddr_t)&fd->fd_done);
}

fdwork(fd, action)
struct fdasd *fd;
int action;
{
	caddr_t thebase;
	int     dawmode;
	int     fdintr();

	fd->fd_tidaw = fd->fd_idaws;
	fd->fd_tccw = fd->fd_ccws;
	fd->fd_trec = fd->fd_recloc;
	thebase = u.u_base;
	if(action == RDDATA)
		dawmode = 0;
	else
		dawmode = 1;
	while(u.u_count != 0) {
		makidaw(dawmode, (caddr_t *)fd->fd_tidaw, thebase, BSIZE);
		if(fdmakerec(fd))
			break;
		fdmakecp(fd, action);
		thebase += BSIZE;
		u.u_count -= BSIZE;
                fd->fd_tidaw++;
                fd->fd_trec++;
	}
	fd->fd_done = 0;
	fd->fd_error = fd->fd_errct = 0;
        sio(fd->fd_devaddr, &fd->fd_ccws[0], fdintr, (int)fd);
	while(fd->fd_done == 0)
		sleep((caddr_t)&fd->fd_done, DSKPRI);
	u.u_error = fd->fd_error;
	return;
}

fdmakecp(fd, rorw)
struct fdasd *fd;
int rorw;
{
	ccw_t *tccw;

	tccw = fd->fd_tccw;
/*
 * First ccw and previous chain bit
 */
	if(tccw != fd->fd_ccws) {
		(tccw - 1)->cc_cc = 1;
		if(fd->fd_trec->fd_cc == (fd->fd_trec - 1)->fd_cc)
			tccw->cc_cmd = SEEKHEAD;
		else
			tccw->cc_cmd = SEEK;
	}
        else
                tccw->cc_cmd = SEEK;
	tccw += 4;
/*
 * Fifth ccw
 */
	tccw->cc_cmd = rorw;
	tccw->cc_cc = 0;
	fd->fd_tccw = tccw + 1;
	return;
}

fdmakerec(fd)
struct fdasd *fd;
{
	struct  fdskdesc *dd;
        int     rec, tmp;

	dd = fd->fd_desc;
	if(u.u_offset >= fd->fd_maxadd)
		return(-1);
        tmp = u.u_offset / BSIZE;

        rec = (tmp % dd->fd_blktrk) + 1;
        tmp = (tmp / dd->fd_blktrk) + TRKSKP;
        fd->fd_trec->fd_cc = tmp / dd->fd_trkcyl;
        fd->fd_trec->fd_hh = tmp % dd->fd_trkcyl;
	if(fd->fd_form) {
                fd->fd_trec->fd_r0 = 0;
		u.u_offset += (dd->fd_blktrk * BSIZE);
	}
	else {
                fd->fd_trec->fd_r0 = rec << 8;
		u.u_offset += BSIZE;
	}
        fd->fd_trec->fd_sector = (dd->fd_const + (rec-1)*dd->fd_factor) / dd->fd_divsor; /* RPS */
	if(u.u_offset > fd->fd_maxadd)
		return(-1);
	return(0);
}


fdread(dev)
int dev;
{
	struct fdasd *fd;

	fd = &fdasds[minor(dev)];
	if(u.u_count % BSIZE != 0) {
		u.u_error = EINVAL;
		return;
	}
	fdwork(fd, RDDATA);
}


fdwrite(dev)
int dev;
{
	struct fdasd *fd;

	fd = &fdasds[minor(dev)];
	if(fd->fd_form) {
                if(u.u_count % (BSIZE * fd->fd_desc->fd_blktrk) != 0 ||
                   u.u_count > MAXWRITE) {
                        u.u_error = EINVAL;
                        return;
                }
                fdformtrk(fd);
	}
	else {
                if(u.u_count % BSIZE != 0) {
                        u.u_error = EINVAL;
                        return;
                }
                fdwork(fd, WRTDATA);
	}
}


fdintr(fd, csw, sense)
struct fdasd *fd;
csw_t *csw;
char *sense;
{
        int retry;
 
        if(csw->cs_uc) {
		printf("Unit check in the Fast disk driver\n");
		printf("Csw = 0x%8x %8x\n", csw->cs_dblw);
		printf("Sense = `");
		for(retry=0; retry<24; retry++)
			printf("%2x", sense[retry]);
		printf("'\n");
		retry = 0;
		if ((BUSOUT || ENVDATA) && !PERMERR) {
			if (fd->fd_errct++ == 0)
				retry = 1;
		} else if ((EQCHK || DATACHK || OVERRUN) && !PERMERR) {
			if (fd->fd_errct++ < 10)
				retry = 1;
		}
		if (retry) {
			sio(fd->fd_devaddr, &fd->fd_ccws[0], fdintr, (int)fd);
			return;
		}
		fd->fd_error = EIO;
                wakeup((caddr_t)&fd->fd_done);
		fd->fd_done = 1;
                return;
        }
        if(csw->cs_de) {
                wakeup((caddr_t)&fd->fd_done);
		fd->fd_done = 1;
		return;
	}
	/*
	 * Some unknown error
	 */
        fd->fd_error = ENXIO;
        wakeup((caddr_t)&fd->fd_done);
        fd->fd_done = 1;
	return;
}

/*
 * Ioctl for the fast disk driver.
 * Set writing for either meek or formatting writes
 * ARGSUSED
 */
fdioctl(dev, cmd, argp, flag)
int dev, cmd, flag;
caddr_t argp;
{
	struct fdasd *fd;

	fd = &fdasds[minor(dev)];
	switch(cmd) {
	case FASTFORM:
		fd->fd_form = 1;
		fdskel(fd);
		suword(argp, BSIZE * fd->fd_desc->fd_blktrk);
		break;
	case FASTNORM:
		fd->fd_form = 0;
	        fdskel(fd);
		break;
	case FASTSIZE:
		suword(argp, (int) (fd->fd_maxadd / BSIZE));
		suword(argp+4, fd->fd_desc->fd_blktrk);
		break;
	default:
		u.u_error = ENOTTY;
		break;
	}
	return;
}


/*
 * Set up skeleton channel programs.
 * For details, see GA26-1592, p. 70.
 */

fdskel(fd)
struct fdasd *fd;
{
	register i, j;
	int last;
	ccw_t *ccwp;
	struct cnt *cntp;
	struct idaws *idawp;

	if(fd->fd_form) {
                            /* no '+1' here because of full track rule. */
		last = ((MAXWRITE / BSIZE) / fd->fd_desc->fd_blktrk);
                ccwp = fd->fd_ccws;
                cntp = fd->fd_cnts;
                idawp = fd->fd_idaws;
                for (i = 0; i < last; i++) {
                        ccwp->cc_dblw = 0;
                        ccwp->cc_cc = 1;
                        ccwp->cc_addr = (int)&fd->fd_recloc[i].fd_bb;
                        ccwp->cc_count = 6;
                        ccwp++;
                        ccwp->cc_dblw = 0;
                        ccwp->cc_cc = 1;
                        ccwp->cc_cmd = SRCHIDE;
                        ccwp->cc_addr = (int)&fd->fd_recloc[i].fd_cc;
                        ccwp->cc_count = 5;
                        ccwp++;
                        ccwp->cc_dblw = 0;
                        ccwp->cc_cc = 1;
                        ccwp->cc_cmd = TIC;
                        ccwp->cc_addr = (int)(ccwp-1);
                        ccwp++;
                        /* write count, key, data */
			for(j=0; j<fd->fd_desc->fd_blktrk; j++) {
                                cntp->fd_count[3] = BSIZE;
                                ccwp->cc_dblw = 0;
				ccwp->cc_cd = 1;
				ccwp->cc_cmd = WRTCKD;
                                ccwp->cc_addr = (int)cntp;
                                ccwp->cc_count = 8;
				ccwp++;
				cntp++;
                                ccwp->cc_cc = 1;
				ccwp->cc_cmd = WRTCKD;
                                ccwp->cc_addr = (int)idawp;
                                ccwp->cc_count = BSIZE;
                                ccwp->cc_ida = 1;
                                idawp++;
				ccwp++;
		        }
		}
		(ccwp - 1)->cc_cc = 0;
	}
	else {
                for (i = 0; i < NCP; i++) {
                        ccwp = &fd->fd_ccws[i*CPSIZE];
                        ccwp->cc_dblw = 0;
                        ccwp->cc_cc = 1;
                        ccwp->cc_addr = (int)&fd->fd_recloc[i].fd_bb;
                        ccwp->cc_count = 6;
                        ccwp++;
                        ccwp->cc_dblw = 0;
                        ccwp->cc_cc = 1;
                        ccwp->cc_cmd = SETSCTR;
                        ccwp->cc_addr = (int)&fd->fd_recloc[i].fd_sector;
                        ccwp->cc_count = 1;
                        ccwp++;
                        ccwp->cc_dblw = 0;
                        ccwp->cc_cc = 1;
                        ccwp->cc_cmd = SRCHIDE;
                        ccwp->cc_addr = (int)&fd->fd_recloc[i].fd_cc;
                        ccwp->cc_count = 5;
                        ccwp++;
                        ccwp->cc_dblw = 0;
                        ccwp->cc_cc = 1;
                        ccwp->cc_cmd = TIC;
                        ccwp->cc_addr = (int)(ccwp-1);
                        ccwp++;
                        /* read or write */
                        ccwp->cc_dblw = 0;
                        ccwp->cc_addr = (int)&fd->fd_idaws[i];
                        ccwp->cc_count = BSIZE;
                        ccwp->cc_ida = 1;
		}
	}
	return;
}

/*
 * Write a complete track using formatted writes.
 * This too is ugly
 */
fdformtrk(fd)
struct fdasd *fd;
{
	struct cnt *cp;
	caddr_t    thebase;
	ccw_t      *ccwp;
	register   i;
	int        fdintr();

	fd->fd_tidaw = fd->fd_idaws;
	fd->fd_trec = fd->fd_recloc;
	ccwp = fd->fd_ccws;
	thebase = u.u_base;
	cp = fd->fd_cnts;
	while(u.u_count != 0) {
                if(fdmakerec(fd)) {
                        u.u_error = EINVAL;
                        return;
                }
                if(ccwp != fd->fd_ccws) {
                        (ccwp - 1)->cc_cc = 1;
                        if(fd->fd_trec->fd_cc == (fd->fd_trec - 1)->fd_cc)
                                ccwp->cc_cmd = SEEKHEAD;
                        else
                                ccwp->cc_cmd = SEEK;
                }
                else
                        ccwp->cc_cmd = SEEK;
                ccwp += 3;
		/*
		 * The writes
		 */
                for(i=0; i<fd->fd_desc->fd_blktrk; i++) {
                        makidaw(1, (caddr_t *)fd->fd_tidaw, thebase, BSIZE);
                        cp->fd_count[0] = fd->fd_trec->fd_cc;
                        cp->fd_count[1] = fd->fd_trec->fd_hh;
                        cp->fd_count[2] = (i+1) << 8;
                        thebase += BSIZE;
                        ccwp += 2;
                        fd->fd_tidaw++;
                        cp++;
                }
		u.u_count -= (BSIZE * fd->fd_desc->fd_blktrk);
                fd->fd_trec++;
	}
	(ccwp - 1)->cc_cc = 0;
	fd->fd_done = 0;
	fd->fd_error = fd->fd_errct = 0;
        sio(fd->fd_devaddr, &fd->fd_ccws[0], fdintr, (int)fd);
	while(fd->fd_done == 0)
		sleep((caddr_t)&fd->fd_done, DSKPRI);
	u.u_error = fd->fd_error;
	return;
}

