/*
 * routines for "memory files"
 * these are like tmp files in that they reside in memory
 * but in this case they're in the process's address space,
 * meaning no system intervention is required
 */

#include <stdio.h>
#include <stdioerr.h>

#define BYTEMASK 0377
#define _IOMREAD (_IOMEM+_IOREAD)
#define _IOMWRT  (_IOMEM+_IOWRT)

static struct _membuf *_getmbuf();
static int ioeof();
char *malloc();

/*
 * currently the filename is unused, however it will be useful
 * for debugging purposes
 */

FILE *memopen(filename, access)
char *filename;
char *access;{

	register FILE *fp;

	fp = _findiop();
	if (*access == 'r')
		fp->_flag = _IOMREAD;
	else
		if (*access == 'w' || *access == 'a')
			fp->_flag = _IOMWRT;
	else
	        return(_stdioerr(EMODE, "memopen", filename));
	fp->_base = NULL;
	fp->_cnt = 0;
	fp->_next = NULL;
	fp->_file = fp - stdin;
	fp->_filename = filename;
	return(fp);
	}

memclose(fp)
FILE *fp;{

	register struct _membuf *p, *q;

	if (fp == NULL)
		return(_stdioerr(EFPNULL, "memclose"));
	if ((fp->_flag&_IOMEM) == 0)
		return(_stdioerr(ENOTMEM, "memclose", fp->_filename));
	for (p = fp->_next ; p != NULL; ) {
		q = p->_mnext;
		free((char *) p);
		p = q;
		}
	fp->_flag = 0;
	fp->_base = NULL;
	fp->_next = NULL;
	fp->_cnt = 0;
	return(0);
	}

FILE *memreopen(fp, access)
register FILE *fp;
char *access;{

	if (fp == NULL)
		return(_stdioerr(EFPNULL, "memreopen"));
	if ((fp->_flag&_IOMEM) == 0)
		return(_stdioerr(ENOTMEM, "memreopen", fp->_filename));
	_mflush(fp);
	switch(*access) {
		case 'r':
			fp->_flag = _IOMREAD;
			if (fp->_base != NULL) {
				fp->_next = (struct _membuf *) fp->_base;
				fp->_ptr = fp->_next->_mbuf;
				fp->_cnt = fp->_next->_mcnt;
				}
			return(fp);

		case 'w':
			fp->_flag = _IOMWRT;
			if (fp->_base != NULL) {
				fp->_next = (struct _membuf *) fp->_base;
			        fp->_ptr = fp->_next->_mbuf;
				fp->_cnt = _mBUFSIZ - fp->_next->_mcnt;
				}
			return(fp);

		case 'a':
			fp->_flag = _IOMWRT;
			_mfseek(fp, 0, 2);
			return(fp);

		default:
			return(_stdioerr(EMODE, "memreopen", fp->_filename));
		}
	}

/*
 * get the next buffer for reading
 */

int _mfilbuf(fp)
register FILE *fp;{

	if (fp == NULL)
		return(_stdioerr(EFPNULL, "_mfilbuf"));
	if ((fp->_flag&_IOMREAD) == 0) {
		fp->_flag |= _IOERR;
		return(_stdioerr(EBADUSE, NULL, fp->_filename, "reading"));
		}
	if (fp->_base==NULL || fp->_next->_mnext==NULL)
		return(ioeof(fp));
	fp->_next = fp->_next->_mnext;
	fp->_ptr = fp->_next->_mbuf;
	fp->_cnt = fp->_next->_mcnt-1;
	return(*fp->_ptr++ & BYTEMASK);
	}

/*
 * flush output buffer
 */

int _mflsbuf(c, fp)
int c;
register FILE *fp;{

	if (fp == NULL)
		return(_stdioerr(EFPNULL, "_mflsbuf"));
	if ((fp->_flag&_IOMWRT) == 0) {
		fp->_flag |= _IOERR;
		return(_stdioerr(EBADUSE, NULL, fp->_filename, "writing"));
		}
	if (fp->_base == NULL) {
		if ((fp->_base = (char *) _getmbuf()) == NULL) {
			fp->_flag |= _IOERR;
			return(EOF);
			}
		fp->_next = (struct _membuf *) fp->_base;
		}
	else {
		fp->_next->_mcnt = fp->_ptr - fp->_next->_mbuf;
		if (fp->_next->_mnext == NULL) {
		        if ((fp->_next->_mnext = _getmbuf()) == NULL) {
				fp->_flag |= _IOERR;
			        return(EOF);
				}
			}
		fp->_next = fp->_next->_mnext;
		}
	fp->_ptr = fp->_next->_mbuf;
	fp->_cnt = _mBUFSIZ-1;
	*fp->_ptr++ = c;
	return(c);
	}

static struct _membuf *_getmbuf() {

	register struct _membuf *p;

        if ((p = (struct _membuf *) malloc((unsigned) MBUFSIZ)) == NULL) {
		_stdioerr(ENOMEM, "_getmbuf");
		return(NULL);
		}
	p->_mcnt = 0;
	p->_mnext = NULL;
	return(p);
	}

static int ioeof(fp)
register FILE *fp;{

	fp->_flag |= _IOEOF;
	return(EOF);
	}

_mflush(fp) 
register FILE *fp;{

	register int len;

	if ((fp->_flag&_IOMWRT) == 0)
		return(_stdioerr(EBADUSE, "fflush", fp->_filename, "writing"));
	if (fp->_base != NULL) {
		fp->_flag &= ~_IOEOF;
		len = fp->_ptr - fp->_next->_mbuf;
		if (len > fp->_next->_mcnt)
			fp->_next->_mcnt = len;
		}
	}

_mftell(fp)
register FILE *fp;{

	register int count;
	register struct _membuf *p;

	count = 0;
	p = (struct _membuf *) fp->_base;
	if (p == NULL) return(0);
	while (p->_mnext != NULL) {
		count += _mBUFSIZ;
		p = p->_mnext;
		}
	return(count + p->_mcnt);
	}

/*
 * seeking is a real pain
 * negative offsets from current position are especially expensive
 */

_mfseek(fp, offset, ptrname)
register FILE *fp;
int offset, ptrname;{

	register struct _membuf *p;

	_mflush(fp);
	if (fp->_base == NULL) {
		if ((fp->_base = (char *) _getmbuf()) == NULL) {
			fp->_flag |= _IOERR;
			return(EOF);
			}
		fp->_next = (struct _membuf *) fp->_base;
		}
	switch(ptrname) {
		case 0:
			p = (struct _membuf *) fp->_base;
			break;

		case 1:
			if (offset < 0) {
				offset = _mftell(fp)+offset;
				p = (struct _membuf *) fp->_base;
				}
			else {
				p = fp->_next;
				offset += p->_mcnt;
				}
			break;

		case 2:
			p = fp->_next;
			while (p->_mnext != NULL) p = p->_mnext;
			offset += p->_mcnt;
			break;

		default:
			fp->_flag |= _IOERR;
			return(_stdioerr(EINVAL, "fseek"));
		}
	while (offset >= _mBUFSIZ) {
		if (p->_mnext == NULL) {
			if ((p->_mnext = _getmbuf()) == NULL) {
				fp->_flag |= _IOERR;
				return(EOF);
				}
			p->_mcnt = _mBUFSIZ;
			}
		p = p->_mnext;
		offset -= _mBUFSIZ;
		}
	if (offset > p->_mcnt)
		p->_mcnt = offset;
	fp->_next = p;
	fp->_ptr = &p->_mbuf[offset];
	if ((fp->_flag&_IOMREAD) == _IOMREAD)
		fp->_cnt = offset;
	else fp->_cnt = _mBUFSIZ - offset;
	fp->_flag &= ~_IOEOF;
	return(0);
	}

_mfread(ptr, size, count, fp)
char *ptr;
int size, count;
register FILE *fp; {

	register int nbytes;

	if ((fp->_flag&_IOMREAD) == 0)
		return(_stdioerr(EBADUSE, "fread", fp->_filename, "reading"));
	if (fp->_base == NULL)
		return(ioeof(fp));
	if (fp->_flag&_IOEOF)
		return(_stdioerr(EPASTEOF, "fread", fp->_filename));
	nbytes = size*count;
	while (nbytes > fp->_cnt) {
		_move(fp->_ptr, ptr, fp->_cnt);
		nbytes -= fp->_cnt;
		ptr += fp->_cnt;
	        if (fp->_next->_mnext == NULL) {
			fp->_flag |= _IOEOF;
			fp->_cnt = 0;
			return((size*count - nbytes) / size);
			}
		fp->_next = fp->_next->_mnext;
		fp->_ptr = fp->_next->_mbuf;
		fp->_cnt = fp->_next->_mcnt;
                }
	_move(fp->_ptr, ptr, nbytes);
	fp->_ptr += nbytes;
	fp->_cnt -= nbytes;
	return(count);
	}

_mfwrite(ptr, size, count, fp)
char *ptr;
int size, count;
register FILE *fp; {

	register int nbytes;
	extern int errno;

	if ((fp->_flag&_IOMWRT) == 0)
		return(_stdioerr(EBADUSE, "fwrite", fp->_filename, "writing"));
	nbytes = size*count;
	if (fp->_base == NULL) {
		if ((fp->_base = (char *) _getmbuf()) == NULL) {
			fp->_flag |= _IOERR;
			return(EOF);
			}
	        fp->_next = (struct _membuf *) fp->_base;
		}
	while (nbytes > fp->_cnt) {
		_move(ptr, fp->_ptr, fp->_cnt);
		if (fp->_next->_mnext == NULL) {
			if ((fp->_next->_mnext = _getmbuf()) == NULL) {
                                fp->_flag |= _IOERR;
				_stdioerr(errno, "fwrite", "writing", fp->_filename);
				return((size*count - nbytes) / count);
                                }
			fp->_next = fp->_next->_mnext;
			fp->_ptr = fp->_next->_mbuf;
			fp->_cnt = _mBUFSIZ-1;
			}
		else {
			fp->_next = fp->_next->_mnext;
			fp->_ptr = fp->_next->_mbuf;
			fp->_cnt = _mBUFSIZ-1;
			}
		}
	_move(ptr, fp->_ptr, nbytes);
	fp->_ptr += nbytes;
	fp->_cnt -= nbytes;
	if (fp->_ptr > &fp->_next->_mbuf[fp->_next->_mcnt])
		fp->_next->_mcnt = fp->_ptr - fp->_next->_mbuf;
	return(count);
	}

_mrew(fp)
register FILE *fp;{

	if (fp->_base == NULL) return;
	_mflush(fp);
	fp->_next = (struct _membuf *) fp->_base;
	fp->_ptr = fp->_next->_mbuf;
	if ((fp->_flag&_IOMREAD) != _IOMREAD)
		fp->_cnt = fp->_next->_mcnt;
	else fp->_cnt = _mBUFSIZ;
	fp->_flag &= ~_IOEOF;
	return;
	}
