/*
 * a common routine for printf's that does dirty work
 *
 * _doprnt is called from printf (and fprintf, and sprintf analogously).
 * When printf is called the variables to be printf'ed are placed
 * on the stack. The argument ap to _doprnt points to the stack where
 * those arguments were put. Fiddling with ap will allow one to access
 * the variables be they characters, character pointers, integers, or
 * whatever.
 *
 * _doprnt uses putc to do all of its output. The io pointer iop
 * is stdout for printf, the io pointer passed to fprintf, or a "string
 * file" pointer for sprintf.
 *
 * _doprnt scans the fmt string looking for a percent %.
 * When one is found, the width and precision values are gotten.
 * A leading minus '-' or zero '0' is noted and a star '*' for the
 * value is handled.
 * The scanning is done by incrementing fmt, looking ahead by
 * referencing fmt+1, and using the char c as an intermediate variable
 * when convenient.
 *
 * A big switch is used to branch to the code for each format letter.
 * The char array 'buff' is filled up either forwards or backwards
 * (backwards for the integer formats u, d, o, and x)
 * and put out at the end. The pointers p and q are used to point to buff.
 * When buff is ready to be output, q points to the leftmost character
 * and p points ONE BEYOND the rightmost one. The padding with blanks
 * or zeroes at the right or the left is done at the end for all formats.
 */

#include <stdio.h>
#include <stdioerr.h>
#include <ctype.h>

#define CHARSIZE 4
#define INTSIZE  4
#define DOUBSIZE 8
#define BUFFSIZE 512
#define OBFSIZE  128
#define FORWARDS 1
#define BACKWARDS 2

#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))

_doprnt(fmt,ap,iop)
register char *fmt, *ap;
FILE *iop;
{
	char c, padchar;
	char buff[BUFFSIZE];
	char *r, *cap;
	char *ecase(), *fcase();
	extern char _sobuf[];
	register char *p, *q;
	int i, s, t, shift, exp, mode, rightadj,
            width, prec, npad, minimal;
	int *iap;
	int buffnull = 0;
	char *oldbuff;
	double *dap;
	long li, ls;
	long *lap;

	if ((iop->_flag & _IONBF) && (stdout->_flag & (_IORW | _IOWRT))) {
		fflush(stdout);
		oldbuff = stdout->_base;
		buffnull = 1;
		setbuf(stdout, NULL);
		setbuf(iop, _sobuf);
	}
	while (c = *fmt++) {
		if (c != '%') {
			putc(c,iop);
			continue;
		}
		rightadj = 1;
		if (*fmt == '-') {
			rightadj = 0;
			++fmt;
		}
		padchar = ' ';
		if (*fmt == '0') {
                        padchar = '0';
			++fmt;
		}
		if (*fmt == '*') {
			iap = (int *) ap;
			width = *iap;
			ap += INTSIZE;
			++fmt;
		} else {
                        width = 0;
                        while (isdigit(c = *fmt++))
                                width = 10*width+(c-'0');
			--fmt;
		}
		minimal = 1;
		prec = -1;
		if (*fmt == '.') {
			++fmt;
			if (*fmt == '0' &&          /* see the f format */
                            (isdigit(*(fmt+1)) || *(fmt+1) == '*')) {
				minimal = 0;
			        ++fmt;
			}
                        if (*fmt == '*') {
				iap = (int *) ap;
                                prec = *iap;
                                ap += INTSIZE;
			        ++fmt;
                        } else {
                                prec = 0;
                                while (isdigit(c = *fmt++))
                                        prec = 10*prec+(c-'0');
				--fmt;
			}
		}
		/*
		 * the next character is the conversion letter.
		 */
		c = *fmt++;
		if (c == 'h')
			c = *fmt++;
		else if (c == 'l') {
			c = *fmt++;
			if (c == 'd' || c == 'o' || c == 'x')
				c += 'A'-'a';
		}
		p = q = buff;
		cap = ap+3;
		iap = (int *) ap;

		switch(c) {
                case 'r':
                        iap = (int *) *iap;
                        fmt = (char *) *iap++;
                        ap = (char *) iap;
                        continue;

                case 'u':
                        if (*iap < 0) {
                                li = (unsigned) *iap;
                                *iap = li/10;
                                *p++ = li%10 + '0';
                        }

                case 'd':
                        ap += INTSIZE;
                        if ((i = *iap) > 0)
                                i = -i;
                        while (s = -(i/10)) {
                                *p++ = '0'-(i+s*10);
                                i = -s;
                        }
                        *p++ = '0'-i;
                        if (*iap < 0 && c != 'u')
                                *p++ = '-';
                        mode = BACKWARDS;
                        break;

                case 'D':
                        lap = (long *) ap;
                        ap += INTSIZE+INTSIZE;
                        if ((li = *lap) > 0)
                                 li = -li;
                        while (ls = -(li/10)) {
                                t = li+(((ls<<2)+ls)<<1);
                                *p++ = '0'-t;
                                li = -ls;
                        }
                        *p++ = '0'-((int) li);
                        if (*lap < 0)
                                *p++ = '-';
                        mode = BACKWARDS;
                        break;

                case 'x':
                        shift = 4;
                        if (rightadj)
                                padchar = '0';

                case 'o':
                        if (c == 'o')
                                shift = 3;
                        i = *iap;
                        ap += INTSIZE;
                        while (s = (i >> shift)) {
                                t = i-(s << shift);
                                if (t >= 10)
                                        *p++ = t-10+'A';
                                else
                                        *p++ = t+'0';
                                i = s;
                        }
                        if (i >= 10)
                                *p++ = i-10+'A';
                        else
                                *p++ = i+'0';
                        mode = BACKWARDS;
                        break;

                case 'X':
                        shift = 4;
                        if (rightadj)
                                padchar = '0';

                case 'O':
                        if (c == 'O')
                                shift = 3;
                        lap = (long *) ap;
                        li = *lap;
                        ap += INTSIZE+INTSIZE;
                        while (ls = (li >> shift)) {
                                t = li-(ls << shift);
                                if (t >= 10)
                                        *p++ = t-10+'A';
                                else
                                        *p++ = t+'0';
                                li = ls;
                        }
                        if ((t = li) >= 10)
                                *p++ = t-10+'A';
                        else
                                *p++ = t+'0';
                        mode = BACKWARDS;
                        break;

                case 'c':
                        *p++ = *cap;
                        ap += CHARSIZE;
                        mode = FORWARDS;
                        break;

                case 's':
                        q = (char *) *iap;
			if (q == NULL)
				_stdioerr(EPRTNULL, NULL, iop->_filename);
			p = q + strlen(q);
			if (prec >= 0 && p-q > prec)
				p = q + prec;
                        ap += INTSIZE;
                        mode = FORWARDS;
                        break;

                case 'g':
                        if (prec == -1)
                                prec = 6;
                        dap = (double *) ap;
                        ap += DOUBSIZE;
			exp = ftoa(*dap, buff+BUFFSIZE/2, prec, 0);
			if (exp > -3 && exp < prec) {
			        q = fcase(*dap, buff+BUFFSIZE/2, prec, minimal, 1);
			        p = q + strlen(q);
			} else {
	                        q = ecase(*dap, buff+2, prec, minimal);
	                        p = q + strlen(q);
			}
			mode = FORWARDS;
			break;

                case 'f':
                        if (prec == -1)
                                prec = 6;
                        dap = (double *) ap;
                        ap += DOUBSIZE;
                        q = fcase(*dap, buff+BUFFSIZE/2, prec, minimal, 0);
			p = q + strlen(q);
                        mode = FORWARDS;
                        break;

                case 'e':
                        if (prec == -1)
                                prec = 6;
                        dap = (double *) ap;
                        ap += DOUBSIZE;
		        q = ecase(*dap, buff+BUFFSIZE/2, prec, minimal);
			p = q + strlen(q);
                        mode = FORWARDS;
                        break;

                default:
                        putc(c,iop);
                        continue;
		}  /* end switch */

		npad = width-(p-q);
		if (rightadj)
			for (i = 1; i <= npad; ++i)
				putc(padchar,iop);
		if (mode == FORWARDS)
			for (r = q; r < p; r++)
				putc(*r,iop);
		else
			for (r = p-1; r >= q; --r)
				putc(*r,iop);
		if (!rightadj)
			for (i = 1; i <= npad; ++i)
				putc(padchar,iop);
	}  /* end while */
	if (buffnull) {
		fflush(iop);
		setbuf(iop, NULL);
                setbuf(stdout, oldbuff);
	}
}

/*
 * The f format is particularily troublesome.
 *
 * If the precision is specified with a leading zero (6.02f or 6.00f)
 * then the format should be columnar oriented. There should always
 * be the same number of digits after the decimal point (zero padded if
 * needed) and there should always be a digit to the left of the decimal
 * point (0.450 rather than .450).
 *
 * If there is no leading zero on the precision these rules are
 * relaxed and the value should be output with the least possible number
 * of characters. The variable minimal is used to flag this kind of format.
 *
 * These two kinds of formatting have the same beginning.
 * The minimal kind ends by chopping extra characters left and right and
 * the columnar format ends by padding with zeros on the right.
 */

static char *
fcase(num, buff, prec, minimal, relprec)
double num;
char *buff;
int prec, minimal, relprec;
{
	int exp, i, minus;
	register char *p, *q;
	char *r;
	char c;

	p = q = buff;
        exp = ftoa(num, p+1, prec, relprec)+1;
        if (exp > 0) {
                /*
                 * the exponent is positive and there are
                 * digits in the buffer to be put on the
                 * left of the decimal point.
                 */
                if (*(p+1) == '-')
                        *p++ = '-';
                for (i = 1; i <= exp; ++i)
                        if (c = *(p+1))
                                *p++ = c;
                        else
                                *p++ = '0';
                *p++ = '.';
        } else {
                /*
                 * the exponent is <= 0 and
                 * we need to put zeroes to the right
                 * of the decimal and a zero to the left.
                 */
                minus = 0;
                if (*(q+1) == '-') {
                        q++;
                        minus = 1;
                }
                for (i = -1; i >= exp; --i)
                        *q-- = '0';
                *q-- = '.';
                p = q+2;
                *q-- = '0';
                if (minus)
                        *q-- = '-';
                ++q;
        }
        /*
         * an invariant:
         * p now points to the char at the right of the
         * decimal point and there is at least one digit
         * (maybe zero) to the right of the decimal point.
         */
        i = 1;
        while (i <= prec && *p) {           /* get at most prec */
                ++p;                        /* digits to the left */
                ++i;                        /* of the decimal point */
        }
        if (minimal) {
                while (*(p-1) == '0')       /* chop the */
                        --p;                /* right side */
                if (*(p-1) == '.')
                        --p;
                if (*q == '0')              /* chop the */
                        ++q;                /* left side */
                else if (*q == '-' &&
                         *(q+1) == '0')
                        *++q = '-';
                if (q == p)                 /* if nothing left */
                        *--q = '0';         /* make it 0 */
                else if (*q == '-' && q+1 == p)
                        *q = '0';
        } else {
                while (i <= prec) {         /* pad with */
                        *p++ = '0';         /* trailing zeroes */
                        ++i;
                }
                if (*q == '-') {           /* check for */
                        r = q+1;           /* minus zero */
                        while (r < p &&
                         (*r == '0' || *r == '.'))
                                ++r;
                        if (r >= p)
                                ++q;
                }
        }
	*p++ = '\0';
	return(q);
}

static char *
ecase(num, buff, prec, minimal)
double num;
char *buff;
int prec, minimal;
{
	int exp, i;
	register char *p, *q;
	char c;

	p = q = buff;
        exp = ftoa(num,p+1,prec,1);
        if (*(p+1) == '-')
                *p++ = '-';
        /*
         * use two steps to move a character down
         * to insure the correct evaluation order.
         * lint complained about *p++ = *(p+1);
         */
        c = *(p+1);
        *p++ = c;
        *p++ = '.';
        i = 1;
        while (i <= prec && *p) {
                ++p;
                ++i;
        }
        if (minimal)
                while (*(p-1) == '0' || *(p-1) == '.')
                        --p;
        else
                while (i <= prec) {
                        *p++ = '0';
                        ++i;
                }
        if (!(minimal && exp == 0)) {
                *p++ = 'e';
                if (exp < 0) {
                        *p++ = '-';
                        exp = -exp;
                } else if (!minimal)
                        *p++ = '+';
                i = exp/10;
                if (!(minimal && i == 0))
                        *p++ = i+'0';
                *p++ = exp%10+'0';
        }
	*p++ = '\0';
	return(q);
}

#define NPOWS 169
#define POWONE 75

extern double pow10[];

static  union {                  /* This is a kludge. */
	long lng;
	double flt;
} max = 0x7fffffffffffffff;      /* Lint doesn't like this.  Too bad. */

/*
 * ftoa is called from ecase and fcase.
 * It does some rounding, fills up a buffer with digits (and maybe a minus
 * sign), and returns an exponent.
 * The exponent is designed for the e format.
 * If x were 51.63, ftoa would fill the buffer with 5163 and return an
 * exponent of 1. The f format will have to add 1 to the exponent
 * to get the number of digits to the right of the decimal point.
 * The code for the g, f, and e formats deals with putting a decimal point
 * at the right place within the buffer and positioning things properly.
 */

#define MAXPREC 17

static
ftoa(x, p, prec, relprec)
double x;
char *p;
int prec, relprec;
{
	int firstpow, lastpow;
	int exp;
	int i;
	double s, tpow;
	char c;
	char *pp;

	pp = p;
	if (x < 0.0) {
		*p++ = '-';
		x = -x;
	}
	if (x < pow10[NPOWS - 1]) {    /* x is zero, or close enough */
		*pp++ = '0';
		*pp = '\0';
		return(0);
	}
	firstpow = findpow(x);
	lastpow = prec + (relprec ? firstpow + 2 : POWONE + 1);
	if (lastpow > NPOWS)
		lastpow = NPOWS;
	else {
		if (max.flt - 0.5 * pow10[lastpow - 1] < x)  /* rounding would cause overflow */
			x = max.flt;
	        else {                 /* round, then adjust first, last pow if necesary */
		        x += 0.5 * pow10[lastpow - 1];
			if (firstpow && x > pow10[firstpow - 1]) {
				firstpow--;
				if (relprec)
					lastpow--;
			}
		}
	}
	if (lastpow > firstpow + MAXPREC)
		lastpow = firstpow + MAXPREC;
	exp = POWONE - firstpow;
	/*
	 * At this point we can assert that:
	 * pow10[firstpow] <= x <= pow10[firstpow - 1]
	 * and that the number of digits we want to put into
	 * the buffer <= lastpow - firstpow.
	 * Now we can actually do the conversion.
	 */
	s = 0.0;
	i = firstpow;
	if (x >= 7 * pow10[0]) {  /* overflow would occur below */
		*p++ = '7';
		x -= 7 * pow10[0];
		i++;
	}
	while (s < x && i < lastpow) {
		/*
		 * Assert: s = sum from j = 0 to i - firstpow - 1 of
		 *         (buf[j] - '0') * pow10[firstpow + j]
		 * and s <= x < s + pow10[i - 1]
		 */
		c = '0';
		tpow = pow10[i];
		while ((s += tpow) <= x && c < '9') /* overflow might occur here but for if test above */
			c++;
		s -= tpow;
		*p++ = c;
		i++;
	}
	*p = '\0';
	return(exp);
}

/*
 * Find the largest power of 10 in x;
 * i.e. return i such that pow10[i] <= x <= pow10[i-1].
 *
 * Entry assertion:  x >= pow10[NPOWS - 1]
 * (If this does not hold, the routine is not gauranteed.)
 */
static
findpow(x)
double x;
{
	int i, low, high;

	if (x >= pow10[0])
		return(0);
	low = 0;
	high = NPOWS - 1;
	while ((high - low) > 1) {
		/* Assertion: pow10[high] <= x <= pow10[low] */
		i = (high + low) / 2;
		if (x < pow10[i])
			low = i;
		else
			high = i;
	}
	return(high);
}
