/*
 *	OPCTL
 *
 *		LPD control functions
 *
 *	Robert Elz, Melb Uni, Feb 1980.
 *	(based upon an earlier version by Kenj McDonell)
 *
 *	Usage:
 *		opctl -function [-d dev] [ job ]...
 *
 *	job can be:
 *		a decimal number (the spool job number reported by opr/opq)
 *		'X'		 (all my jobs - must be only job #)
 *		'XX' (su's only) (all jobs)
 *
 *	Options:
 *		-k	kick the daemon - generally for ungluing anything stuck
 *		-a	abort the current print file on the printer spec'd
 *		-z	discard a print job (current or not) completely
 *		-r	reprint the current print file on the printer spec'd
 *		-q	send the job on the printer spec'd back to the queue
 *		-p	pause printing on the printer spec'd
 *		-c	restart after a -p
 *		-s	stop printing on printer spec'd after current file
 *
 *		-d dev	specify a printer (by class or device name)
 *			(printers can also specified by implication from
 *			 a job number)
 *
 *	Bugs:
 *		Some output almost always gets lost with -p / -c and -s
 *		functions (or by altering comments, gets printed twice)
 *		(see below)
 *		With a job number of 'X', some lf* files may be left
 *		around, when they shouldn't be (iff the user has printed
 *		files that he doesn't own, and links were used)
 *		- su's can clean up by opctl -z XX when all is quiet
 */

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include "local.h"
#include "spool.h"

extern	char	lpd[];

int	maxchild;		/* to keep 'merge' happy */

int	cmd;			/* the function to perform */
int	kflag;			/* '-k' specified */
int	devflg;			/* '-d' specified */
int	allmine;		/* job number 'X' - all my jobs */
int	alljobs;		/* job number 'XX' - (su only) - all jobs */

char	device[CLSIZE];		/* the arg with '-d': printer class */

extern	struct pr prt[];
extern	struct pr *lastpr;

int jcnt;			/* the number of 'job#' args given */
int arg[200];			/* the actual args (200 should be plenty) */

int prcount;			/* the number of printers that match our '-d' */

int timedout;			/* != 0 --> no reply from lpd */
int sigok;			/* != 0 --> lpd reply affirmative */

FILE	*dfd;			/* open spool directory - for searches */

struct pr *active();
char	  *rindex();

main(argc, argv)
char **argv;
{
	PR pr;
	register struct qjob *jp;
	register char *p, c;
	PR apr;
	register i;

	while (--argc > 0)
		if ( *(p = *++argv) == '-' ) while (c = *++p) switch (c) {

		case 'a':	setcom(LPD_ABT);	break;
		case 'c':	setcom(LPD_CONT);	break;
		case 'p':	setcom(LPD_PAUS);	break;
		case 'q':	setcom(LPD_REQ);	break;
		case 'r':	setcom(LPD_REP);	break;
		case 's':	setcom(LPD_STOP);	break;
		case 'z':	setcom(LPD_KILL);	break;

		case 'k':	if (cmd != 0)
					usage();
				kflag++;
							break;

		case 'd':	if (p[1] != '\0')
					p++;
				else if (argc > 1 && argv[1][0] != '-')
					argc--, p = *++argv;
				else
					p++;
				if (!devflg) {
					STRCPM(device, p);
					devflg++;
				}
				if (*p == '\0')
					p--;
				while (p[1] != '\0')
					p++;
							break;

		default:	usage();
							break;

		} else
			break; /* from the outer while loop */

	jcnt = 0;
	while (argc-- > 0) {
		if (*p != 'X' && !isdigit(*p))
			usage();
		if (*p == 'X')
			if (getuid() || p[1] != 'X')
				allmine++;
			else
				alljobs++;
		else if ((arg[jcnt] = atoi(p)) == 0)
			fprintf(stderr, "Opctl: bad job number: %s\n", p);
		else
			jcnt++;
		p = *++argv;
	}

	if ( (allmine || alljobs) && jcnt > 0)
		usage();

	if (kflag && devflg) {
		cmd = LPD_KICK;
		kflag = 0;
	}
	if (kflag)
		kick();

	if (cmd == 0)
		usage();

	merge();
	getactive();

	prcount = 0;
	if (devflg) for (printers)
		if (thispr(pr)) {
			prcount++;
			apr = pr;
		}
	if (devflg && prcount == 0) {
		fprintf(stderr, "Opctl: no '%s' printers available\n", device);
		exit(1);
	}

	if (chdir(lpd) < 0) {
		fprintf(stderr, "Opctl: can't chdir to SPOOLDIR\n");
		exit(1);
	}
	if ((dfd = fopen(lpd, "r")) == NULL) {
		fprintf(stderr, "Opctl: can't open SPOOLDIR\n");
		exit(1);
	}

	if (alljobs || allmine)
		dothelot();

	if (jcnt == 0) {
		if (prcount > 1)
			fprintf(stderr, "Opctl: ambiguous device specification\n");
		else if (prcount == 0)
			fprintf(stderr, "Opctl: ctl functions reguire -d or job#\n");
		else
			killit(apr);
		exit(prcount == 1);
	}

	for (i = 0; i < jcnt; i++)
		if (pr = active(arg[i]))
			if (devflg && !thispr(pr))
				fprintf
				(	stderr,
					"Opctl: %d now printing on %s (/dev/%s: %s)\n"
					,pr->pjob
					,pr->class
					,pr->prnm
					,pr->locn
				);
			else
				killit(pr);
		else
			ctl(arg[i]);
	exit(0);
}

usage()
{
	fprintf(stderr, "Usage: opctl [ -k ] [ -acpqrsz ] [ -dDEV ] [ JOB ]..\n");
	exit(1);
}

setcom(c)
{
	if (kflag || cmd != 0)
		usage();

	cmd = c;
}

killit(pr)
PR pr;
{
	struct	lpdcmd	lpdcmd;
	register fd;
	register i;
	int	lpdpid;
	int	timeout(), succeed(), fail();

		/*
		 * for the 'zap' function, delete all associated files
		 */
	if (cmd == LPD_KILL && pr->pjob)
		ctl(pr->pjob);

#ifdef	LOCK
	if ((fd = open(LPDLOCK, 0)) >= 0) {
		close(fd);
#else
	if (access(LOCKFILE, 0) < 0) {
#endif
		fprintf(stderr, "Opctl: lpd not active\n");
		exit(1);
	}

	if ((fd = open(LOCKFILE, 0)) < 0) {
		fprintf(stderr, "Opctl: lockfile open error\n");
		exit(1);
	}

	for (i = 0; i < 3; i++)
		if (read(fd, &lpdpid, sizeof lpdpid) != sizeof lpdpid
				|| lpdpid == 0)
			sleep(1);
		else
			break;

	if (i >= 3) {
		fprintf(stderr, "Opctl: can't read lockfile\n");
		exit(1);
	}
	close(fd);

	for (i = 0; i < 5; i++)
		if (access(KILLFILE, 0) >= 0)
			sleep(1);

	if (access(KILLFILE, 0) >= 0) {
  /***		this is the safe way ...
		fprintf(stderr, "Opctl: previous control function pending\n");
		exit(1);
 ***		and this is where even fools fear to tread ... ***/
		unlink(KILLFILE);
	}

	umask(0);

	signal(SIGINT, SIG_IGN);
	signal(SIGHUP, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);

	if ((fd = creat(KILLFILE, 0444)) < 0) {
		fprintf(stderr, "Opctl: cannot create communications file\n");
		exit(1);
	}

	lpdcmd.xjob = getpid();
	lpdcmd.cjob = pr->pjob;
	lpdcmd.xfn  = cmd;
	STRCPN(lpdcmd.xname, pr->prnm);

	if (write(fd, &lpdcmd, sizeof lpdcmd) != sizeof lpdcmd) {
		close(fd);
		unlink(KILLFILE);
		fprintf(stderr, "Opctl: write error on commun file\n");
		exit(1);
	}

	signal(SIGALRM, timeout);
	signal(FAIL, fail);
	signal(SUCCEED, succeed);
	timedout = sigok = 0;
	alarm(5);
	if (kill(lpdpid, CSIG) < 0) {
		alarm(0);
		unlink(KILLFILE);
		fprintf(stderr, "Opctl: can't signal LPD\n");
		exit(1);
	}
	pause();
	signal(SUCCEED, SIG_IGN);
	signal(FAIL, SIG_IGN);
	signal(SIGALRM, SIG_IGN);
	alarm(0);

	if (!sigok) {
		unlink(KILLFILE);
		if (timedout) {
			fprintf(stderr, "Opctl: no response from LPD\n");
			exit(1);
		}
		fprintf(stderr, "Opctl: cmd to job %d refused\n", pr->pjob);
	}
}

timeout()
{
	timedout++;
}

succeed()
{
	sigok++;
}

fail()
{
}

ctl(n)
{
	struct direct dir;
	char	nbuf[DIRSIZ+1];
	register fnd = 0;
	register char *p;

	if (cmd != LPD_KILL) {
		fprintf(stderr, "Opctl: Job %d not active\n", n);
		return;
	}
	nbuf[DIRSIZ] = '\0';
	rewind(dfd);
	while (fread(&dir, sizeof dir, 1, dfd) == 1) {
		if (dir.d_ino == 0)
			continue;
		if (dir.d_name[1] != 'f')
			continue;
		STRCPM(nbuf, dir.d_name);
		if ((p = rindex(nbuf, '.')) == NULL)
			continue;
		if (n != atoi(p+3))
			continue;
		if (nbuf[0] == 'd')
			fnd++;
		unlink(nbuf);
	}

	if (!fnd)
		fprintf(stderr, "Opctl: Job %d not found\n", n);
}

kick()
{
	register fd, i;
	int lpdpid;

#ifdef	LOCK
	if ((fd = open(LPDLOCK, 0)) < 0) {
#else
	if (access(LOCKFILE, 0) >= 0) {
#endif
		for (i = 0; i < 3; i++)
			if ((fd = open(LOCKFILE, 0)) < 0)
				sleep(1);
			else
				break;
		if (fd < 0) {
			execle(DAEMON, "LPD", 0, 0);
			exit(1);
		}
		for (i = 0; i < 3; i++)
			if (read(fd, &lpdpid, sizeof lpdpid) == sizeof lpdpid)
				break;
			else
				sleep(1);
		close(fd);
		if (i >= 3 || lpdpid == 0 || kill(lpdpid, MERGE) < 0) {
			fprintf(stderr, "Opctl: can't signal LPD\n");
			exit(1);
		}

		exit(0);
	}
#ifdef	LOCK
	close(fd);
#endif

	execle(DAEMON, "LPD", 0, 0);
	fprintf(stderr, "Opctl: can't exec LPD\n");
	exit(1);
}

dothelot()
{
	PR pr;
	PR apr;
	register char *p;
	register me = getuid();
	long pos;
	char nbuf[DIRSIZ+1];
	struct stat statb;
	struct direct dir;

	nbuf[DIRSIZ] = '\0';
	rewind(dfd);

	while (fread(&dir, sizeof dir, 1, dfd) == 1) {
		if (dir.d_ino == 0)
			continue;
		STRCPM(nbuf, dir.d_name);
		if (nbuf[1] != 'f' || (p = rindex(nbuf, '.')) == NULL)
			continue;
		if (!alljobs) {
			if (stat(nbuf, &statb) < 0 || statb.st_uid != me)
				continue;
		}
		if (devflg) {
			*p = '\0';
			for (printers)
				if (thispr(pr) && checkpr(pr, &nbuf[2]))
					break;
			if (pr >= lastpr) {
				fprintf(stderr, "Opctl: Can't locate printer for %d\n", atoi(p+3));
				continue;
			}
			*p = '.';
		}
		if (apr = active(atoi(p+3))) {
			pos = ftell(dfd);
			killit(apr);
			fseek(dfd, pos, 0);
			continue;
		}
		if (cmd != LPD_KILL)		/* ignore others for non active */
			continue;
		unlink(nbuf);
	}
}
