/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)DoRoute.c	1.9 84/10/18
*/

/*
**	Applies routing algorithm to each element of address in 'HdrDest',
**	and for each outgoing link found builds a new destination address
**	in 'HdrDest' and calls '*funcp' with a new source address as arg.
**
**	If any address error is encountered,
**	'*errfuncp' is called with the offending node name as argument,
**	and an error reason in nlp->nl_name.
**
**	Returns true if '*funcp' was called, otherwise false.
*/

#include	"global.h"

#include	"address.h"
#include	"debug.h"
#include	"header.h"
#include	"state.h"

#include	"route.h"


/*
**	Declare structures for making multicast disjoint sets.
*/

typedef struct NodeEl *	NodeEl_p;

typedef struct NodeEl
{
	NodeEl_p	ne_next;
	char *		ne_addr;
}
			NodeEl;

typedef struct
{
	NodeEl_p	ns_next;
	NodeLink	ns_link;
	short		ns_count;
}
			NodeSet;



bool
DoRoute(source, link, nlp, funcp, errfuncp)
	char *		source;		/* The source address */
	char *		link;		/* The incoming link address */
	NodeLink *	nlp;		/* Place for details of outgoing link */
	void		(*funcp)();	/* Function to send message on link */
	void		(*errfuncp)();	/* Address error function */
{
	register char *	cp1;
	register char *	cp2;
	register int	i;
	register int	j;
	bool		val;
	NodeLink	snl;

	Trace4(1, "DoRoute link \"%s\", source \"%s\", home \"%s\"", link, source, HomeNode);

	if ( RouteBase == NULLSTR && !ReadRoute() )
	{
		/*
		**	No route file!
		*/

		nlp->nl_name = "no routing table";

		(*errfuncp)(source, true);

		return false;
	}

	if
	(
		link != NULLSTR
		&&
		!FindNode(link, pt_msg, &snl)
	)
		Fatal2("Unknown link \"%s\" in DoRoute", link);

	switch ( HdrDest[0] )
	{
	case ATYP_BROADCAST:
	{
		/*
		**	Extended Reverse Path Forwarding Algorithm:
		**
		**	If this message has come via the link on the
		**	shortest path back to the source, then it should
		**	be propagated on all those links for which a bit
		**	is set in the broadcast table, and which is a
		**	member of the target domain.
		*/

		bool	freecp1;
		int	primdom;	/* Target domain index */

		if ( HdrDest[1] == DOMAIN_SEP )
		{
			NodeLink	dom_link;

			if ( HdrDest[2] == ATYP_BROADCAST )
				primdom = LINK_N_A;
			else
			{
				if ( !FindDomain(&HdrDest[2], &dom_link) )
				{
					/*
					**	Unknown domain!
					*/

					nlp->nl_name = dom_link.nl_name;	/* Export fail reason */

					(*errfuncp)(HdrDest, false);
					return false;
				}

				primdom = dom_link.nl_domind;
			}
		}

		if ( strcmp(source, HomeNode) == STREQUAL )
		{
			/*
			**	Start of broadcast.
			**	Set boolean for each link
			**	on its own shortest path
			**	which is also in the target domain.
			*/

			if ( HdrDest[1] != DOMAIN_SEP )
				primdom = RT_NODE(NodeCount-1)->ne_primary;

			cp1 = Malloc(1+LinkCount/8);
			j = 0;	/* Offset */
			freecp1 = true;

			for ( i = 0 ; i < LinkCount ; i++ )
				if ( RT_NODE(RT_LINK(i)->le_index)->ne_shortest[(int)pt_msg] == i )
					cp1[i/8] |= (1<<(i%8));
				else
					cp1[i/8] &= ~(1<<(i%8));
		}
		else
		{
			cp1 = newstr(source);	/* Might get modified by "FindAddress()" */

			if ( !FindAddress(cp1, nlp) )
			{
				/*
				**	Unknown source!
				*/

				(*errfuncp)(source, true);

				free(cp1);
				return false;
			}

			Trace3
			(
				2,
				"Broadcast source \"%s\" (address \"%s\")",
				nlp->nl_name,
				cp1
			);

			if ( strcmp(nlp->nl_name, link) != STREQUAL )
			{
				/*
				**	Didn't come the right way, so drop it.
				*/

				free(cp1);
				return false;
			}

			if ( HdrDest[1] != DOMAIN_SEP )
			{
				if ( strchr(cp1, DOMAIN_SEP) )
				{
					/*
					**	This one has escaped from its primary domain, so drop it!
					*/

					free(cp1);
					return false;
				}

				primdom = RT_NODE(nlp->nl_index)->ne_primary;
			}

			free(cp1);
			freecp1 = false;

			j = nlp->nl_index * LinkCount;
			cp1 = &ForwTable[j/8];
			j %= 8;	/* Offset */
		}

		for ( val = false, i = 0 ; i < LinkCount ; i++, j++ )
			if ( cp1[j/8] & (1<<(j%8)) )
			{
				(void)GetLink(i, nlp);

				if
				(
					primdom != LINK_N_A
					&&
					nlp->nl_domind != LINK_N_A
				)
				{
					register int	k;

					/*
					**	Test link lies on route to target domain.
					*/

					k = primdom * LinkCount + i;

					if ( !(DomForwTable[k/8] & (1<<(k%8))) )
						continue;
				}

				Trace2
				(
					2,
					"Broadcast link \"%s\"",
					nlp->nl_name
				);

				if
				(
					link != NULLSTR
					&&
					(cp2 = AdjustSource(source, snl.nl_index, nlp->nl_index)) != NULLSTR
				)
				{
					(*funcp)(cp2);
					free(cp2);
				}
				else
					(*funcp)(source);

				val = true;
			}

		if ( freecp1 )
			free(cp1);

		return val;
	}

	case ATYP_MULTICAST:
	{
		/*
		**	Need to make disjoint sets of nodes with common
		**	shortest paths from here.
		*/

		register NodeSet *	nsp;
		register NodeEl_p	nep;
		register NodeSet *	link_dests;

		link_dests = nsp = (NodeSet *)Malloc(LinkCount * sizeof(NodeSet));

		for ( i = 0 ; i < LinkCount ; i++, nsp++ )
			nsp->ns_count = 0;

		cp1 = HdrDest;

		do
		{
			*cp1++ = ATYP_MULTICAST;

			if ( (cp2 = strchr(cp1, ATYP_MULTICAST)) == NULLSTR )
			{
				if ( cp1 == &HdrDest[1] )
				{
					nlp->nl_name = "Bad multicast address";
					(*errfuncp)(cp1, false);
					return false;
				}
			}
			else
				*cp2 = '\0';

			nep = Talloc(NodeEl);

			nep->ne_addr = newstr(cp1);

			if ( !FindAddress(nep->ne_addr, nlp) )
			{
				/*
				**	Unknown destination!
				*/

				free(nep->ne_addr);
				free((char *)nep);

				(*errfuncp)(cp1, false);

				continue;
			}

			if ( (nsp = &link_dests[nlp->nl_link])->ns_count == 0 )
			{
				nsp->ns_link = *nlp;
				nep->ne_next = (NodeEl_p)0;
			}
			else
				nep->ne_next = nsp->ns_next;

			nsp->ns_next = nep;
			nsp->ns_count++;
		}
			while ( (cp1 = cp2) != NULLSTR );
		
		Trace2(2, "Multi-cast address \"%s\"", HdrDest);

		/*
		**	Make an address set in HdrDest for each outgoing link.
		*/

		val = false;

		for ( nsp = link_dests, i = 0 ; i < LinkCount ; i++, nsp++ )
		{
			if ( nsp->ns_count == 0 )
				continue;

			if ( nsp->ns_count == 1 )
			{
				nep = nsp->ns_next;
				(void)strcpy(HdrDest, nep->ne_addr);
				free(nep->ne_addr);
				free((char *)nep);
			}
			else
			for
			(
				cp1 = HdrDest ;		/* NB: length of sub-set <= original */
				(nep = nsp->ns_next) != (NodeEl_p)0 ;
				nsp->ns_next = nep->ne_next,
				free((char *)nep)
			)
			{
				*cp1++ = ATYP_MULTICAST;
				cp1 = strcpyend(cp1, nep->ne_addr);
				free(nep->ne_addr);
			}

			*nlp = nsp->ns_link;

			Trace3
			(
				2,
				"Multi-cast subset \"%s\" on link \"%s\"",
				HdrDest,
				nlp->nl_name
			);

			if
			(
				link != NULLSTR
				&&
				(cp2 = AdjustSource(source, snl.nl_index, nlp->nl_index)) != NULLSTR
			)
			{
				(*funcp)(cp2);
				free(cp2);
			}
			else
				(*funcp)(source);

			val = true;
		}

		free((char *)link_dests);

		return val;
	}

	case ATYP_EXPLICIT:
		if ( (cp1 = strchr(&HdrDest[1], ATYP_EXPLICIT)) == NULLSTR )
		{
			nlp->nl_name = "Bad explicit address";
			(*errfuncp)(&HdrDest[1], false);
			return false;
		}
		
		*cp1 = '\0';

		if ( !FindAddress(&HdrDest[1], nlp) )
		{
			/*
			**	Unknown destination!
			*/

			(*errfuncp)(&HdrDest[1], false);

			*cp1 = ATYP_EXPLICIT;

			return false;
		}

		if ( (cp2 = &HdrDest[strlen(HdrDest)]) != cp1 )
		{
			/*
			**	HdrDest truncated by "FindAddress()".
			*/

			*cp2++ = ATYP_EXPLICIT;
			(void)strcpy(cp2, &cp1[1]);
		}
		else
			*cp1 = ATYP_EXPLICIT;

		break;

	default:
		if ( !FindAddress(HdrDest, nlp) )
		{
			/*
			**	Unknown destination!
			*/

			(*errfuncp)(HdrDest, false);

			return false;
		}
	}

	Trace3(2, "Route for \"%s\" via \"%s\"", HdrDest, nlp->nl_name);

	if
	(
		link != NULLSTR
		&&
		(cp2 = AdjustSource(source, snl.nl_index, nlp->nl_index)) != NULLSTR
	)
	{
		(*funcp)(cp2);
		free(cp2);
	}
	else
		(*funcp)(source);

	return true;
}
