/*
**	Written by Doug Kingston, Centrum voor Wiskunde en Informatica, NL
*/

/*
 *				acslogin.c
 *
 *	Run setuid to root as the shell for ACSNET daemon logins.
 *	It makes various security checks before execing the shell
 *	with a "-" to insure that .profile is run.  .profile may
 *	make additional security or access checks if it wishes.
 *	We always invoke /bin/sh as -ACSNET[hostname] if the
 *	security checks are passed.
 */
#include <stdio.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "global.h"

extern	char	*getenv();

char	execname[64];

main (argc, argv)
int	argc;
char	**argv;
{
	struct passwd *pw;
	struct group *gr;
	struct stat sb;
	int	acsnetuid;
	char	*user;
	char	*home;

	if ((pw = getpwnam(ACSNETNAME)) == (struct passwd *)0) {
		fprintf(stderr, "ACSNET owner account (%s) not in /etc/passwd.\n", ACSNETNAME);
		exit(9);
	}
	acsnetuid = pw->pw_uid;

	if ((pw = getpwuid(getuid())) == (struct passwd *)0) {
		fprintf(stderr, "No user for uid %d!\n", getuid());
		exit(9);
	}

	if ((user = getenv("USER")) == NULL) {
		fprintf(stderr, "USER environment variable not set.\n");
		exit(9);
	}

	/*
	 *  We verify the USER string is correct.  This means that
	 *  we can write a shell script to do authentication
	 *  and session setup, and it can be sure it is doing so
	 *  based on good data.
	 */
	if (strcmp(pw->pw_name, user)) {
		fprintf(stderr, "USER/uid mismatch.\n");
		exit(9);
	}

	/*
	 *  More checking, make sure we have a proper $HOME.
	 */
	if ((home = getenv("HOME")) == NULL) {
		fprintf(stderr, "HOME environment variable not set.\n");
		exit(9);
	}

	if (strcmp(pw->pw_dir, home)) {
		fprintf(stderr, "Inappropriate HOME environment variable.\n");
		exit(9);
	}

	/* Let's just be sure... */
	if (chdir(home) < 0) {
		perror("home");
		exit(9);
	}

	/*
	 *  The primary security check:  Is this user in the ACSNET group?
	 *  Two possible ways to qualify.  a) the real GID of this process
	 *  is that of the ACSNET group, or b) this user is explicitly
	 *  listed in the /etc/group file.
	 */
	if ((gr = getgrnam(ACSNETGRP)) == (struct group *)NULL) {
		fprintf(stderr, "ACSNET group ('%s') is unknown.\n", ACSNETGRP);
		exit(9);
	}
	if (getgid() != gr->gr_gid) {
		for (; *gr->gr_mem; gr->gr_mem++)
			if (strcmp(*gr->gr_mem, user) == 0)
				break;
		if(*gr->gr_mem == NULL) {
			fprintf(stderr, "Not a member of ACSNET group.\n");
			exit(9);
		}
	}

	/*
	 *  Assuming the USER is correct, we check to see that a
	 *  ".profile" file exists in the user's home directory, and
	 *  that it is owned by a "trusted" user (0 or ACSNETUID).
	 *  Given that we pass these tests, we exec /bin/sh with
	 *  fd0 redirected from the login file.  Remember that when
	 *  we finally go to run the daemon, we will have to
	 *  redirect the input to /dev/tty
	 */
	if (stat(".profile", &sb) < 0) {
		perror(".profile");
		exit(9);
	}
	if (sb.st_uid != 0 && sb.st_uid != acsnetuid) {
		fprintf(stderr, ".profile owned by inappropriate user (uid %d).\n", sb.st_uid);
		exit(9);
	}
	if (setuid(acsnetuid) < 0) {
		perror("setuid");
		exit(9);
	}

	sprintf(execname, "-ACSNET[%s]", user);
	execl("/bin/sh", execname, 0);
	perror("/bin/sh");
	exit(8);
}
